<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| Controllers
|--------------------------------------------------------------------------
*/
use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\UploadController;
use App\Http\Controllers\Api\AnalyticsController;
use App\Http\Controllers\Api\AuthCustomerController;
use App\Http\Controllers\Api\ProfileController;
use App\Http\Controllers\Api\PaymentController;
use App\Http\Controllers\Api\ShippingController;
use App\Http\Controllers\Api\CustomerOrderController;
use App\Http\Controllers\Api\DeliveryController;
use App\Http\Controllers\Api\BiteshipController;
use App\Http\Controllers\Api\BiteshipWebhookController;
use App\Http\Controllers\Api\OrderInvoiceController;
use App\Http\Controllers\Api\AddressSearchController;

/* Master Data */
use App\Http\Controllers\Api\{
    BankAccountController,
    BannerController,
    BranchController,
    CategoryController,
    ProductController,
    ProductVariantController,
    ProductImageController,
    ProductBranchStockController,
    ProductReviewController,
    VoucherController,
    SettingController,
    BrandController,
};

/* Transaction */
use App\Http\Controllers\Api\{
    CartController,
    OrderController,
    OrderItemController,
    PosSessionController,
};

/* User */
use App\Http\Controllers\Api\{
    UserController,
    UserAddressController,
    UserVoucherController,
    WishlistController,
    CustomerController,
    StaffController,
};

/* Notification & Helper */
use App\Http\Controllers\Api\{
    NotificationController,
    NotificationSubscriptionController,
    OtpVerificationController,
    PasswordResetController,
};

/*
|--------------------------------------------------------------------------
| CORS preflight
|--------------------------------------------------------------------------
*/
Route::options('{any}', function () {
    return response()->json([], 200);
})->where('any', '.*');

/*
|--------------------------------------------------------------------------
| PUBLIC AUTH ROUTES
|--------------------------------------------------------------------------
*/
Route::post('/auth/login', [AuthController::class, 'login']);
Route::post('/auth/register', [AuthController::class, 'register']);
Route::post('/auth/google', [AuthController::class, 'googleLogin']);

/*
|--------------------------------------------------------------------------
| PRODUCT REVIEWS (PUBLIC + AUTH)
|--------------------------------------------------------------------------
| NOTE:
| - /product-reviews/me harus di atas /product-reviews/{productReview}
| - /product-reviews/{productReview} dibatasi hanya angka
|--------------------------------------------------------------------------
*/

/* PUBLIC: list (admin dashboard kamu pakai GET ini) */
Route::get('/product-reviews', [ProductReviewController::class, 'index']);

/* AUTH: my reviews + CRUD */
Route::middleware('auth:sanctum')->group(function () {
    Route::get('/product-reviews/me', [ProductReviewController::class, 'myReviews']);

    Route::post('/product-reviews', [ProductReviewController::class, 'store']);
    Route::put('/product-reviews/{productReview}', [ProductReviewController::class, 'update']);
    Route::delete('/product-reviews/{productReview}', [ProductReviewController::class, 'destroy']);
});

/* STAFF/ADMIN: reply (INI YANG FRONTEND BUTUH) */
Route::middleware(['auth:sanctum', 'staff'])->group(function () {
    Route::patch('/product-reviews/{productReview}/reply', [ProductReviewController::class, 'reply']);
});

/* PUBLIC: show by id (paling bawah + hanya angka) */
Route::get('/product-reviews/{productReview}', [ProductReviewController::class, 'show'])
    ->whereNumber('productReview');

/*
|--------------------------------------------------------------------------
| AUTHENTICATED ROUTES (GENERAL)
|--------------------------------------------------------------------------
*/
Route::middleware('auth:sanctum')->group(function () {
    /* Auth */
    Route::get('/auth/me', [AuthController::class, 'me']);
    Route::post('/auth/logout', [AuthController::class, 'logout']);

    /* Upload */
    Route::post('/uploads', [UploadController::class, 'store']);

    /* Analytics */
    Route::get('/analytics', [AnalyticsController::class, 'index']);

    /*
    |--------------------------------------------------------------------------
    | ADMIN AREA (STAFF ONLY)
    |--------------------------------------------------------------------------
    */
    Route::prefix('admin')->middleware('staff')->group(function () {
        /* Super Admin & Admin only */
        Route::middleware('roleAny:super_admin,admin')->group(function () {
            /* Staff */
            Route::get('/staff', [StaffController::class, 'index']);
            Route::post('/staff', [StaffController::class, 'store']);
            Route::get('/staff/{user}', [StaffController::class, 'show']);
            Route::put('/staff/{user}', [StaffController::class, 'update']);
            Route::delete('/staff/{user}', [StaffController::class, 'destroy']);

            /* OPTIONAL: route reply versi admin prefix (biarin ada kalau kamu mau) */
            Route::patch('/product-reviews/{productReview}/reply', [ProductReviewController::class, 'reply']);

            /* Customers */
            Route::get('/customers', [CustomerController::class, 'index']);
            Route::get('/customers/{id}', [CustomerController::class, 'show']);
        });
    });

    /*
    |--------------------------------------------------------------------------
    | PRODUCT VARIANTS (NESTED)
    |--------------------------------------------------------------------------
    */

    Route::get('/variants', [ProductVariantController::class, 'byIds']);
    Route::prefix('products/{product}')->group(function () {
        Route::get('/variants', [ProductVariantController::class, 'index']);
        Route::post('/variants', [ProductVariantController::class, 'store']);
        Route::put('/variants/{variant}', [ProductVariantController::class, 'update']);
        Route::delete('/variants/{variant}', [ProductVariantController::class, 'destroy']);


    });
});

/*
|--------------------------------------------------------------------------
| API RESOURCES (AUTH REQUIRED)
|--------------------------------------------------------------------------
*/
Route::middleware('auth:sanctum')->group(function () {
    /* Master */
    Route::apiResource('vouchers', VoucherController::class);
    Route::apiResource('bank-accounts', BankAccountController::class);
    Route::apiResource('banners', BannerController::class);
    Route::apiResource('branches', BranchController::class);
    Route::apiResource('categories', CategoryController::class);

    Route::get('/categories', [CategoryController::class, 'index']);
    Route::get('/categories/{id}', [CategoryController::class, 'show']);
    Route::get('/category/{slug}', [CategoryController::class, 'showBySlug']);

    Route::post('/categories', [CategoryController::class, 'store']);
    Route::put('/categories/{id}', [CategoryController::class, 'update']);
    Route::delete('/categories/{id}', [CategoryController::class, 'destroy']);

    Route::apiResource('products', ProductController::class);
    Route::apiResource('product-images', ProductImageController::class);
    Route::apiResource('product-branch-stocks', ProductBranchStockController::class);
    Route::apiResource('settings', SettingController::class);

    /* Transactions */
    Route::apiResource('carts', CartController::class);
    Route::apiResource('orders', OrderController::class);
    Route::apiResource('order-items', OrderItemController::class);
    Route::apiResource('pos-sessions', PosSessionController::class);

    /* Users */
    Route::apiResource('users', UserController::class);
    Route::apiResource('user-vouchers', UserVoucherController::class);
    Route::apiResource('wishlists', WishlistController::class);

    /* Notifications & OTP */
    Route::apiResource('notification-subscriptions', NotificationSubscriptionController::class);
    Route::apiResource('otp-verifications', OtpVerificationController::class);
    Route::apiResource('password-resets', PasswordResetController::class);
});

/*
|--------------------------------------------------------------------------
| CURRENT USER
|--------------------------------------------------------------------------
*/
Route::middleware('auth:sanctum')->get('/user', function (Request $request) {
    return $request->user();
});

/*
|--------------------------------------------------------------------------
| PUBLIC (CUSTOMER)
|--------------------------------------------------------------------------
*/
Route::get('/products', [ProductController::class, 'index']);
Route::get('/products/{product}', [ProductController::class, 'show']);
Route::get('/brands', [BrandController::class, 'index']);

Route::post('/customer/loginc', [AuthCustomerController::class, 'loginCustomer']);
Route::post('/logout', [AuthCustomerController::class, 'logout'])->middleware('auth:sanctum');

Route::post('/customer/send-otp', [OtpVerificationController::class, 'sendOtp']);
Route::post('/customer/verify-otp', [OtpVerificationController::class, 'verifyOtp']);
Route::post('/customer/complete-profile', [AuthCustomerController::class, 'completeProfile']);
Route::post('/customer/login-otp', [AuthCustomerController::class, 'loginWithOtp']);
Route::post('/customer/check-password', [AuthCustomerController::class, 'checkPassword']);
Route::post('/customer/send-reset-password-otp', [AuthCustomerController::class, 'sendResetPasswordOtp']);
Route::post('/customer/reset-password', [AuthCustomerController::class, 'resetPassword']);

/*
|--------------------------------------------------------------------------
| PROFILE / ADDRESSES
|--------------------------------------------------------------------------
*/
Route::middleware('auth:sanctum')->group(function () {
    Route::get('/profile', [ProfileController::class, 'show']);
    Route::put('/profile', [ProfileController::class, 'update']);
    Route::put('/profile/password', [ProfileController::class, 'updatePassword']);
    Route::post('/profile/avatar', [ProfileController::class, 'updateAvatar']);

    Route::get('/addresses', [UserAddressController::class, 'index']);
    Route::post('/addresses', [UserAddressController::class, 'store']);
    Route::put('/addresses/{id}', [UserAddressController::class, 'update']);
    Route::delete('/addresses/{id}', [UserAddressController::class, 'destroy']);
    Route::put('/addresses/{id}/primary', [UserAddressController::class, 'setPrimary']);
});

/*
|--------------------------------------------------------------------------
| CART
|--------------------------------------------------------------------------
*/
Route::middleware('auth:sanctum')->group(function () {
    Route::get('/cart', [CartController::class, 'index']);
    Route::post('/cart', [CartController::class, 'store']);
    Route::put('/cart/{cart}', [CartController::class, 'update']);
    Route::delete('/cart/{cart}', [CartController::class, 'destroy']);
});

/*
|--------------------------------------------------------------------------
| PAYMENT
|--------------------------------------------------------------------------
*/
Route::middleware('auth:sanctum')->group(function () {
    Route::post('/payment/midtrans', [PaymentController::class, 'createMidtrans']); 
    Route::get('/orders/{order}/status', [PaymentController::class, 'orderStatus']);
});

Route::post('/midtrans/webhook', [PaymentController::class, 'midtransWebhook']);

Route::middleware('auth:sanctum')->post('/checkout', [ShippingController::class, 'store']);
Route::middleware('auth:sanctum')->get('/my-orders', [CustomerOrderController::class, 'myOrders']);

Route::prefix('shipping')->middleware('auth:sanctum')->group(function () {
    Route::post('rates', [BiteshipController::class, 'rates']);
    Route::post('orders', [BiteshipController::class, 'createOrder']);
    Route::post('orders/{order}/biteship', [BiteshipController::class, 'createOrderFromLocal']);
    Route::get('orders/{orderId}', [BiteshipController::class, 'track']);
});

Route::middleware('auth:sanctum')->group(function () {
    Route::get('/orders/{order}/invoice', [OrderInvoiceController::class, 'invoice']);
    Route::post('/orders/{order}/send-invoice-email', [OrderInvoiceController::class, 'sendInvoiceEmail']);
    Route::patch('/orders/{order}/status', [OrderInvoiceController::class, 'updateStatus']);
});

Route::post('webhooks/biteship', [BiteshipWebhookController::class, 'handle']);

Route::middleware('auth:sanctum')->group(function () {
    Route::get('/address/search', [AddressSearchController::class, 'search']);
});

Route::middleware('auth:sanctum')->group(function () {
    Route::get('/notifications/unread-count', [NotificationController::class, 'unreadCount']);
    Route::patch('/notifications/read-all', [NotificationController::class, 'markAllRead']);
    Route::patch('/notifications/{notification}/read', [NotificationController::class, 'markRead'])
        ->whereNumber('notification');

    Route::apiResource('notifications', NotificationController::class)
        ->where(['notification' => '[0-9]+']);
});

Route::middleware('auth:sanctum')->group(function () {
  Route::prefix('admin')->middleware('staff')->group(function () {
    Route::middleware('roleAny:super_admin,admin,inventory')->group(function () {
      Route::apiResource('brands', BrandController::class);
    });
  });
});

