<?php

namespace App\Services;

use Illuminate\Http\Client\PendingRequest;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class BiteshipService
{
    private function client(): PendingRequest
    {
        $key = trim((string) config('services.biteship.key'));
        $baseUrl = rtrim((string) config('services.biteship.base_url'), '/');

        Log::debug('BITESHIP config loaded', [
            'base_url' => $baseUrl,
            'key_prefix' => substr($key, 0, 12),
            'key_length' => strlen($key),
        ]);

        return Http::acceptJson()
            ->baseUrl($baseUrl)
            ->withHeaders([
                'Authorization' => 'Bearer ' . $key, // ✅ ini yang dipakai
                'Content-Type'  => 'application/json',
            ])
            ->timeout(20);
    }

    public function getRates(array $payload): Response
    {
        $res = $this->client()->post('/v1/rates/couriers', $payload);

        Log::debug('BITESHIP getRates', [
            'status' => $res->status(),
            'body' => $res->json(),
        ]);

        return $res;
    }

    private function baseUrl(): string
    {
        return rtrim(config('services.biteship.base_url', env('BITESHIP_BASE_URL', 'https://api.biteship.com')), '/');
    }

    private function apiKey(): string
    {
        return (string) config('services.biteship.key', env('BITESHIP_API_KEY', ''));
    }

    private function headers(): array
    {
        return [
            'Authorization' => 'Bearer ' . $this->apiKey(),
            'Accept'        => 'application/json',
            'Content-Type'  => 'application/json',
        ];
    }

    public function rates(array $payload)
    {
        $url = $this->baseUrl() . '/v1/rates/couriers';

        return Http::withHeaders($this->headers())
            ->timeout(30)
            ->post($url, $payload);
    }

    public function createOrder(array $payload)
    {
        $url = $this->baseUrl() . '/v1/orders';

        return Http::withHeaders($this->headers())
            ->timeout(30)
            ->post($url, $payload);
    }

    public function track($orderId)
    {
        $url = $this->baseUrl() . '/v1/orders/' . $orderId;

        return Http::withHeaders($this->headers())
            ->timeout(30)
            ->get($url);
    }
}
