<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class StaffController extends Controller
{
    private array $staffRoles = ['super_admin','admin','inventory','cs'];

    public function index(Request $request)
    {
        $q = $request->query('q');
        $perPage = (int) $request->query('per_page', 10);

        $users = User::query()
            ->whereIn('role', $this->staffRoles)
            ->when($q, function ($query) use ($q) {
                $query->where(function ($sub) use ($q) {
                    $sub->where('name', 'like', "%{$q}%")
                        ->orWhere('email', 'like', "%{$q}%")
                        ->orWhere('phone', 'like', "%{$q}%");
                });
            })
            ->orderByDesc('id')
            ->paginate($perPage);

        return response()->json($users);
    }

    public function show(User $user)
    {
        abort_unless(in_array($user->role, $this->staffRoles, true), 404);
        return response()->json($user);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => ['required','string','max:100'],
            'email' => ['required','email','max:100','unique:users,email'],
            'password' => ['required','string','min:8'],
            'phone' => ['nullable','string','max:20'],
            'role' => ['required', Rule::in($this->staffRoles)],
            'branch_id' => ['nullable','exists:branches,id'],
            'is_active' => ['nullable','boolean'],
        ]);

        $data['password'] = Hash::make($data['password']);
        $data['is_active'] = (bool)($data['is_active'] ?? true);

        $user = User::create($data);

        return response()->json($user, 201);
    }

    public function update(Request $request, User $user)
    {
        abort_unless(in_array($user->role, $this->staffRoles, true), 404);

        $data = $request->validate([
            'name' => ['sometimes','string','max:100'],
            'email' => ['sometimes','email','max:100', Rule::unique('users','email')->ignore($user->id)],
            'password' => ['sometimes','string','min:8'],
            'phone' => ['nullable','string','max:20'],
            'role' => ['sometimes', Rule::in($this->staffRoles)],
            'branch_id' => ['nullable','exists:branches,id'],
            'is_active' => ['nullable','boolean'],
        ]);

        if (array_key_exists('password', $data)) {
            $data['password'] = Hash::make($data['password']);
        }

        $user->update($data);

        return response()->json($user);
    }

    public function destroy(User $user)
    {
        abort_unless(in_array($user->role, $this->staffRoles, true), 404);

        if ($user->role === 'super_admin') {
            $count = User::where('role','super_admin')->count();
            if ($count <= 1) {
                return response()->json([
                    'message' => 'Tidak bisa menghapus super admin terakhir.'
                ], 422);
            }
        }

        $user->delete();

        return response()->json(['message' => 'Staff deleted']);
    }
}