<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\UserAddress;

class ShippingController extends Controller
{
    public function store(Request $request)
    {
        $request->validate([
            'shipping_courier' => 'required',
            'shipping_service' => 'required',
            'shipping_cost' => 'required|numeric',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required',
            'items.*.variant_id' => 'required',
            'items.*.product_name' => 'required',
            'items.*.variant_name' => 'required',
            'items.*.quantity' => 'required|numeric|min:1',
            'items.*.price' => 'required|numeric',
            'items.*.hpp' => 'required|numeric',
        ]);

        $user = $request->user();

        $address = UserAddress::where('user_id', $user->id)
            ->where('is_primary', 1)
            ->first();

        if (!$address) {
            return response()->json([
                'message' => 'Alamat utama belum diatur'
            ], 422);
        }

        DB::beginTransaction();

        try {
            $invoice = 'INV-' . date('Ymd') . '-' . strtoupper(Str::random(6));

            $subtotal = collect($request->items)->sum(fn ($i) =>
                $i['price'] * $i['quantity']
            );

            $totalHpp = collect($request->items)->sum(fn ($i) =>
                $i['hpp'] * $i['quantity']
            );

            // === CREATE ORDER (TANPA PAYMENT) ===
            $order = Order::create([
                'user_id' => $user->id,
                'delivery_address' => $address->address_full,
                'delivery_postal_code' => $address->postal_code,
                'delivery_recipient_name' => $address->recipient_name,
                'delivery_phone' => $address->phone_number,
                'invoice_number' => $invoice,
                'transaction_source' => 'online',
                'total_amount' => $subtotal + $request->shipping_cost,
                'total_hpp' => $totalHpp,
                'shipping_cost' => $request->shipping_cost,
                'shipping_courier' => $request->shipping_courier,
                'shipping_service' => $request->shipping_service,
            ]);

            foreach ($request->items as $item) {
                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $item['product_id'],
                    'variant_id' => $item['variant_id'],
                    'product_name' => $item['product_name'],
                    'variant_name' => $item['variant_name'],
                    'quantity' => $item['quantity'],
                    'price' => $item['price'],
                    'hpp' => $item['hpp'],
                ]);
            }

            DB::commit();

            return response()->json([
                'message' => 'Order berhasil dibuat',
                'order' => $order->load('items')
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => 'Checkout gagal',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
