<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class ProfileController extends Controller
{
    public function show(Request $request)
    {
        $user = $request->user();
        $user->avatar_url = $user->avatar_url;

        return response()->json($user);
    }

    public function update(Request $request)
    {
        $data = $request->validate([
            'name'        => 'required|string|max:100',
            'phone'       => 'nullable|string|max:20',
            'birth_date'  => 'nullable|date',
            'gender'      => 'nullable|in:Laki-laki,Perempuan',
        ]);

        $request->user()->update($data);

        return response()->json([
            'message' => 'Profil berhasil diperbarui',
            'user'    => $request->user()
        ]);
    }

    /* ================= AVATAR ================= */
    public function updateAvatar(Request $request)
    {
        $request->validate([
            'avatar' => 'required|image|mimes:jpg,jpeg,png|max:10240', // ✅ 10MB
        ]);

        $user = $request->user();

        // hapus avatar lama
        if ($user->avatar && Storage::disk('public')->exists('avatars/' . $user->avatar)) {
            Storage::disk('public')->delete('avatars/' . $user->avatar);
        }

        $file = $request->file('avatar');
        $filename = 'avatar_' . $user->id . '_' . time() . '.' . $file->getClientOriginalExtension();

        $file->storeAs('avatars', $filename, 'public');

        $user->update([
            'avatar' => $filename
        ]);

        return response()->json([
            'message' => 'Foto profil berhasil diubah',
            'avatar_url' => asset('storage/avatars/' . $filename)
        ]);
    }

    public function updatePassword(Request $request)
    {
        $request->validate([
            'password' => 'required|min:8|confirmed'
        ]);

        $request->user()->update([
            'password' => Hash::make($request->password)
        ]);

        return response()->json([
            'message' => 'Password berhasil diubah'
        ]);
    }
}
