<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductVariant;
use Illuminate\Http\Request;

class ProductVariantController extends Controller
{
    public function index(Product $product)
    {
        return response()->json($product->variants()->latest('id')->get());
    }

   public function store(Request $request, Product $product)
    {
        $data = $request->validate([
            'variant_name' => 'nullable|string|max:200',
            'name' => 'nullable|string|max:200', // optional kalau FE kirim name
            'sku' => 'nullable|string|max:100',
            'price' => 'nullable|numeric',
            'hpp' => 'nullable|numeric',
            'stock' => 'nullable|integer',
            'image_url' => 'nullable|string|max:500',
        ]);

        // kalau DB butuh `name`, isi dari variant_name atau fallback
        $data['name'] = $data['name'] ?? $data['variant_name'] ?? $product->name;

        // kalau Anda masih belum punya timestamps di table, matikan di model atau tambah timestamps
        $variant = $product->variants()->create([
            'name' => $data['name'],
            'sku' => $data['sku'] ?? null,
            'price' => $data['price'] ?? null,
            'hpp' => $data['hpp'] ?? null,
            'stock' => $data['stock'] ?? 0,
            'image_url' => $data['image_url'] ?? null,
        ]);

        return response()->json($variant, 201);
    }

    public function update(Request $request, Product $product, ProductVariant $variant)
    {
        if ($variant->product_id !== $product->id) {
            return response()->json(['message' => 'Variant tidak milik product ini.'], 404);
        }

        $data = $request->validate([
            'variant_name' => 'nullable|string|max:200',
            'name' => 'nullable|string|max:200',
            'sku' => 'nullable|string|max:100',
            'price' => 'nullable|numeric',
            'hpp' => 'nullable|numeric',
            'stock' => 'nullable|integer',
            'image_url' => 'nullable|string|max:500',
        ]);

        if (array_key_exists('variant_name', $data) && !array_key_exists('name', $data)) {
            $data['name'] = $data['variant_name']; // mapping
        }

        unset($data['variant_name']); // biar gak nyasar ke kolom yg gak ada

        $variant->update($data);

        return response()->json($variant);
    }

    public function destroy(Product $product, ProductVariant $variant)
    {
        if ($variant->product_id !== $product->id) {
            return response()->json(['message' => 'Variant tidak milik product ini.'], 404);
        }

        $variant->delete();
        return response()->json(['message' => 'Variant deleted']);
    }
}