<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $q          = trim((string) $request->get('q', ''));
        $categoryId = $request->get('category_id');
        $brand      = $request->get('brand');      // slug brand
        $perPage    = $request->get('per_page');   // optional

        $query = Product::query()
            ->with(['category', 'brand'])
            ->withCount('variants')
            ->withSum('variants as stock_total', 'stock')
            ->withMin('variants as price_min', 'price')
            ->withMax('variants as price_max', 'price')
            ->withMin('variants as hpp_min', 'hpp')
            ->withMax('variants as hpp_max', 'hpp')
            ->latest('id');

        /**
         * 🔍 SEARCH
         */
        if ($q !== '') {
            $query->where(function ($w) use ($q) {
                $w->where('name', 'like', "%{$q}%")
                  ->orWhere('sku', 'like', "%{$q}%");
            });
        }

        /**
         * 📁 CATEGORY FILTER
         */
        if (!empty($categoryId)) {
            $query->where('category_id', $categoryId);
        }

        /**
         * 🏷 BRAND FILTER (by slug)
         */
        if (!empty($brand)) {
            $query->whereHas('brand', function ($b) use ($brand) {
                $b->where('slug', $brand);
            });
        }

        /**
         * ===============================
         * RESPONSE MODE
         * ===============================
         */

        // 👉 ADMIN / LIST PAGE (PAGINATION)
        if (!empty($perPage)) {
            $perPage = max(1, min((int) $perPage, 100));

            return response()->json(
                $query->paginate($perPage)
            );
        }

        // 👉 CATEGORY PAGE / FRONTEND (AMBIL SEMUA)
        return response()->json([
            'data' => $query->get()
        ]);
    }
    public function show(Product $product)
    {
        $product->load('category', 'variants', 'images', 'reviews');
        return response()->json($product);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'category_id' => 'required|exists:categories,id',
            'brand_id' => 'nullable|exists:brands,id',
            'name' => 'required|string|max:200',
            'slug' => 'nullable|string|max:255|unique:products,slug',
            'sku'  => 'nullable|string|max:50|unique:products,sku',
            'description' => 'nullable|string',
            'thumbnail_url' => 'nullable|string',
            'weight_grams' => 'required|integer',
            'is_active' => 'boolean',
        ]);

        // generate slug kalau kosong
        if (empty($data['slug'])) {
            $base = Str::slug($data['name']);
            $slug = $base ?: 'product';

            $i = 2;
            while (Product::where('slug', $slug)->exists()) {
                $slug = $base . '-' . $i;
                $i++;
            }
            $data['slug'] = $slug;
        }

        $data['is_active'] = $data['is_active'] ?? 1;
        $data['brand_id'] = $data['brand_id'] ?? null;

        $product = Product::create($data);
        return response()->json($product, 201);
    }

    public function update(Request $request, Product $product)
    {
        $request->validate([
            'category_id' => 'sometimes|exists:categories,id',
            'name' => 'sometimes|string|max:200',
            'brand_id' => 'sometimes|nullable|exists:brands,id',
            'slug' => 'nullable|string|max:255|unique:products,slug,' . $product->id,
            'sku'  => 'nullable|string|max:50|unique:products,sku,' . $product->id,
            'description' => 'nullable|string',
            'thumbnail_url' => 'nullable|string',
            'weight_grams' => 'sometimes|integer',
            'is_active' => 'boolean',
        ]);

        $product->update($request->all());
        return response()->json($product);
    }

    public function destroy(Product $product)
    {
        $product->delete();
        return response()->json(['message' => 'Product deleted']);
    }
}