<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Midtrans\Config;
use Midtrans\Snap;
use App\Models\Order;

class PaymentController extends Controller
{
    /* ===============================
     | SET MIDTRANS CONFIG
     =============================== */
    private function initMidtrans()
    {
        Config::$serverKey    = config('midtrans.server_key');
        Config::$isProduction = (bool) config('midtrans.is_production');
        Config::$isSanitized  = true;
        Config::$is3ds        = true;
    }

    /* ===============================
     | CREATE SNAP TOKEN MIDTRANS
     =============================== */
    public function createMidtrans(Request $request)
    {
        $request->validate([
            'order_id' => 'required|exists:orders,id',
        ]);

        $order = Order::findOrFail($request->order_id);

        // Cegah token dibuat ulang kalau sudah paid
        if ($order->payment_status === 'paid') {
            return response()->json(['message' => 'Order sudah dibayar'], 400);
        }

        $this->initMidtrans();

        $params = [
            'transaction_details' => [
                'order_id'     => $order->invoice_number,
                'gross_amount' => (int) $order->total_amount,
            ],
            'customer_details' => [
                'first_name' => $order->delivery_recipient_name ?? 'Customer',
                'phone'      => $order->delivery_phone ?? '-',
            ],
        ];

        try {
            $snapToken = Snap::getSnapToken($params);

            $order->update([
                'payment_method' => 'midtrans',
                'payment_status' => 'pending',
            ]);

            return response()->json([
                'snap_token' => $snapToken,
            ]);
        } catch (\Throwable $e) {
            Log::error('MIDTRANS SNAP TOKEN ERROR', [
                'order_id' => $order->id,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'message' => 'Gagal membuat Snap Token'
            ], 500);
        }
    }

    /* ===============================
     | UPLOAD BUKTI TRANSFER MANUAL
     =============================== */
    public function uploadProof(Request $request, $id)
    {
        $request->validate([
            'proof' => 'required|image|mimes:jpg,jpeg,png|max:2048',
        ]);

        $order = Order::findOrFail($id);

        $path = $request->file('proof')->store('payment-proofs', 'public');

        $order->update([
            'payment_method' => 'manual',
            'payment_proof'  => $path,
            'payment_status' => 'awaiting_verification',
        ]);

        return response()->json([
            'message' => 'Bukti pembayaran berhasil diupload',
        ]);
    }

    /* ===============================
     | MIDTRANS WEBHOOK (SECURE)
     =============================== */
    public function midtransWebhook(Request $request)
    {
        $payload = $request->all();
        Log::info('MIDTRANS WEBHOOK RAW', $payload);

        // Validasi minimal payload
        if (
            empty($payload['order_id']) ||
            empty($payload['status_code']) ||
            empty($payload['gross_amount']) ||
            empty($payload['signature_key'])
        ) {
            return response()->json(['message' => 'Invalid payload'], 400);
        }

        // Verify signature
        $serverKey = config('midtrans.server_key');
        $expectedSignature = hash(
            'sha512',
            $payload['order_id'] . $payload['status_code'] . $payload['gross_amount'] . $serverKey
        );

        if ($payload['signature_key'] !== $expectedSignature) {
            Log::warning('MIDTRANS SIGNATURE INVALID', [
                'expected' => $expectedSignature,
                'got' => $payload['signature_key'],
            ]);
            return response()->json(['message' => 'Invalid signature'], 403);
        }

        $order = Order::where('invoice_number', $payload['order_id'])->first();

        if (!$order) {
            return response()->json(['message' => 'Order not found'], 404);
        }

        $transactionStatus = $payload['transaction_status'] ?? null;
        $fraudStatus = $payload['fraud_status'] ?? null;

        if (in_array($transactionStatus, ['settlement', 'capture'])) {
            // kalau capture dan fraud challenge -> bisa pending verifikasi
            if ($fraudStatus === 'challenge') {
                $order->update(['payment_status' => 'pending']);
            } else {
                $order->update([
                    'payment_status' => 'paid',
                    'paid_at' => now(),
                ]);
            }
        } elseif ($transactionStatus === 'pending') {
            $order->update(['payment_status' => 'pending']);
        } elseif (in_array($transactionStatus, ['expire', 'cancel', 'deny'])) {
            $order->update(['payment_status' => 'failed']);
        }

        return response()->json(['success' => true]);
    }
}
