<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Order;

class CustomerOrderController extends Controller
{
    public function myOrders(Request $request)
    {
        $user = $request->user();

        $orders = Order::with(['items.product.images'])
            ->where('user_id', $user->id)
            ->latest()
            ->get()
            ->map(function ($order) {

                $item = $order->items->first();
                $product = $item?->product;

                $imageModel = $product?->images
                    ->where('is_cover', 1)
                    ->first()
                    ?? $product?->images->first();

                // ✅ NORMALIZE PATH (hindari "storage/storage/..." dan handle path yang sudah ada prefix "storage/")
                $imageUrl = null;
                if ($imageModel && !empty($imageModel->path)) {
                    $cleanPath = ltrim(str_replace('storage/', '', $imageModel->path), '/');
                    $imageUrl = asset('storage/' . $cleanPath);
                }

                return [
                    'id' => $order->id,
                    'invoice' => $order->invoice_number,
                    'total' => (float) $order->total_amount,
                    'product_name' => $product?->name ?? '-',

                    // ✅ INI YANG DIPERBAIKI
                    'image' => $imageUrl,

                    // ✅ STATUS CUSTOMER
                    'status' => $this->status($order),
                    'status_label' => $this->statusLabel($order),

                    // ✅ PENTING UNTUK TRACKING
                    'biteship_order_id' => $order->biteship_order_id,
                    'shipping_tracking_number' => $order->shipping_tracking_number,
                    'shipping_courier' => $order->shipping_courier,
                    'shipping_service' => $order->shipping_service,

                    // ✅ kalau kamu sudah tambah field ini di DB
                    'delivery_postal_code' => $order->delivery_postal_code ?? null,

                    // Debug (boleh hapus nanti)
                    'order_status' => $order->order_status,
                    'payment_status' => $order->payment_status,

                    'date' => $order->created_at->format('d F Y'),
                ];
            });

        return response()->json($orders);
    }

    private function status(Order $order)
    {
        if (in_array($order->order_status, [
            'processing', 'shipped', 'delivered', 'cancelled', 'returned'
        ])) {
            return match ($order->order_status) {
                'processing' => 'PROCESSING',
                'shipped' => 'SHIPPED',
                'delivered' => 'COMPLETED',
                'cancelled', 'returned' => 'FAILED',
                default => 'PROCESSING',
            };
        }

        return match ($order->payment_status) {
            'pending' => 'UNPAID',
            'awaiting_verification' => 'VERIFICATION',
            'failed', 'expired' => 'FAILED',
            default => 'UNPAID',
        };
    }

    private function statusLabel(Order $order)
    {
        if (in_array($order->order_status, [
            'processing', 'shipped', 'delivered', 'cancelled', 'returned'
        ])) {
            return match ($order->order_status) {
                'processing' => 'DIPROSES',
                'shipped' => 'DIKIRIM',
                'delivered' => 'SELESAI',
                'cancelled' => 'DIBATALKAN',
                'returned' => 'DIKEMBALIKAN',
                default => 'DIPROSES',
            };
        }

        return match ($order->payment_status) {
            'pending' => 'BELUM BAYAR',
            'awaiting_verification' => 'VERIFIKASI',
            'failed' => 'GAGAL',
            'expired' => 'KADALUARSA',
            default => 'BELUM BAYAR',
        };
    }
}
