<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Services\BiteshipService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;


class BiteshipController extends Controller
{

    private function baseUrl(): string
{
    return rtrim(config('services.biteship.base_url', env('BITESHIP_BASE_URL', 'https://api.biteship.com')), '/');
}

private function apiKey(): string
{
    return (string) config('services.biteship.key', env('BITESHIP_API_KEY', ''));
}

private function biteshipHeaders(): array
{
    return [
        'Authorization' => 'Bearer ' . $this->apiKey(),
        'Accept'        => 'application/json',
        'Content-Type'  => 'application/json',
    ];
}

    public function __construct(private BiteshipService $biteship) {}

    /**
     * POST /api/shipping/rates
     * Body contoh:
     * {
     *   "origin_postal_code":"40111",
     *   "destination_postal_code":"40286",
     *   "couriers":"jne,jnt,sicepat",
     *   "items":[{"name":"Produk A","quantity":1,"weight":1000,"value":50000}]
     * }
     */
    public function rates(Request $request)
    {
        $data = $request->validate([
            'origin_postal_code' => 'required|string',
            'destination_postal_code' => 'required|string',
            'couriers' => 'required|string',
            'items' => 'required|array|min:1',
            'items.*.name' => 'required|string',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.weight' => 'required|integer|min:1',
            'items.*.value' => 'nullable|numeric|min:0',
        ]);

        $res = $this->biteship->rates($data);

        return response()->json($res->json(), $res->status());
    }

    /**
     * POST /api/shipping/orders
     * (kalau kamu mau bikin order langsung dari FE tanpa local order)
     */
    public function createOrder(Request $request)
    {
        $data = $request->validate([
            'shipper_contact_name' => 'required|string',
            'shipper_contact_phone' => 'required|string',
            'shipper_contact_email' => 'nullable|email',

            'shipper_organization' => 'nullable|string',
            'origin_postal_code' => 'required|string',
            'origin_address' => 'required|string',
            'origin_contact_name' => 'required|string',
            'origin_contact_phone' => 'required|string',

            'destination_contact_name' => 'required|string',
            'destination_contact_phone' => 'required|string',
            'destination_contact_email' => 'nullable|email',
            'destination_address' => 'required|string',
            'destination_postal_code' => 'required|string',

            'courier_company' => 'required|string', // jne
            'courier_type' => 'required|string',    // reg
            'courier_insurance' => 'nullable|integer',

            'items' => 'required|array|min:1',
            'items.*.name' => 'required|string',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.weight' => 'required|integer|min:1',
            'items.*.value' => 'required|numeric|min:0',
        ]);

        $res = $this->biteship->createOrder($data);

        return response()->json($res->json(), $res->status());
    }

    /**
     * POST /api/shipping/orders/{order}/biteship
     * ini yang kamu panggil dari admin untuk generate resi (AWB).
     *
     * Body minimal dari FE:
     * {
     *   "company": "jne",
     *   "type": "reg"
     * }
     *
     * OPTIONAL:
     * - weight (gram)
     * - destination_postal_code (kalau order belum simpan)
     */
    public function createOrderFromLocal(Request $request, Order $order)
{
    Log::debug('BITESHIP createOrderFromLocal incoming', [
        'order_id' => $order->id,
        'body' => $request->all(),
    ]);

    if (!empty($order->biteship_order_id) || !empty($order->shipping_tracking_number)) {
        return response()->json([
            'success' => false,
            'message' => 'Biteship order / resi sudah ada untuk order ini.',
        ], 409);
    }

    // ✅ dukung 2 format:
    // 1) { company, type, destination_postal_code, ... }
    // 2) { courier: { company, type, destination_postal_code, ... } }
    $payloadInput = $request->input('courier', $request->all());

    // ✅ VALIDASI: destination_postal_code wajib
    $validator = \Validator::make($payloadInput, [
        'company' => 'required|string',
        'type' => 'required|string',
        'destination_postal_code' => 'required|string|min:5',
        'insurance' => 'nullable|numeric',
        'weight' => 'nullable|numeric|min:1',
    ]);

    if ($validator->fails()) {
        return response()->json([
            'success' => false,
            'message' => 'Validasi gagal.',
            'errors' => $validator->errors(),
            'received' => $payloadInput,
        ], 422);
    }

    $data = $validator->validated();

    // Load items
    $order->load(['items']);
    if ($order->items->count() === 0) {
        return response()->json([
            'success' => false,
            'message' => 'Order tidak punya item. Tidak bisa buat pengiriman.',
        ], 422);
    }

    // ORIGIN config
    $originPostalCode = (string) config('services.biteship.origin_postal_code', '');
    $originContactName = (string) config('services.biteship.origin_contact_name', 'Store');
    $originContactPhone = (string) config('services.biteship.origin_contact_phone', '080000000000');
    $originAddress = (string) config('services.biteship.origin_address', 'Alamat Toko');

    $missingOrigin = [];
    if (!$originPostalCode) $missingOrigin[] = 'BITESHIP_ORIGIN_POSTAL_CODE';
    if (!$originContactName) $missingOrigin[] = 'BITESHIP_ORIGIN_CONTACT_NAME';
    if (!$originContactPhone) $missingOrigin[] = 'BITESHIP_ORIGIN_CONTACT_PHONE';
    if (!$originAddress) $missingOrigin[] = 'BITESHIP_ORIGIN_ADDRESS';

    if (!empty($missingOrigin)) {
        return response()->json([
            'success' => false,
            'message' => 'Konfigurasi origin Biteship belum lengkap.',
            'missing' => $missingOrigin,
        ], 500);
    }

    // ✅ Normalisasi courier type (contoh Anteraja)
    $company = strtolower(trim($data['company']));
    $type = strtolower(trim($data['type']));

    // Banyak checkout nyimpen "reguler", sedangkan Biteship sering pakai "reg"
    $typeMap = [
        'reguler' => 'reg',
        'regular' => 'reg',
    ];
    if (isset($typeMap[$type])) $type = $typeMap[$type];

    // items
    $items = [];
    foreach ($order->items as $it) {
        $items[] = [
            'name'     => $it->product_name ?? 'Item',
            'quantity' => max(1, (int) ($it->quantity ?? 1)),
            'value'    => max(0, (int) ($it->price ?? 0)),
            'weight'   => 200,
        ];
    }

    $weight = (int) ($data['weight'] ?? 1000);

    $payload = [
        'reference_id' => $order->invoice_number ?? ('ORDER-' . $order->id),

        'origin_contact_name'  => $originContactName,
        'origin_contact_phone' => $originContactPhone,
        'origin_address'       => $originAddress,
        'origin_postal_code'   => $originPostalCode,

        'destination_contact_name'  => $order->delivery_recipient_name ?: ($order->customer_name_manual ?: 'Customer'),
        'destination_contact_phone' => $order->delivery_phone ?: '080000000000',
        'destination_address'       => $order->delivery_address ?: '-',
        'destination_postal_code'   => $data['destination_postal_code'],

        'courier_company' => $company,
        'courier_type'    => $type,

        'delivery_type' => 'now',
        'items' => $items,
        'weight' => $weight,
        'insurance' => (int) ($data['insurance'] ?? 0),
        'order_note' => 'Order #' . ($order->invoice_number ?? $order->id),
    ];

    Log::debug('BITESHIP createOrderFromLocal payload', [
        'order_id' => $order->id,
        'payload'  => $payload,
    ]);

    $url = $this->baseUrl() . '/v1/orders';

    $res = Http::withHeaders($this->biteshipHeaders())
        ->timeout(30)
        ->post($url, $payload);

    Log::debug('BITESHIP createOrderFromLocal response', [
        'status' => $res->status(),
        'body'   => $res->json(),
    ]);

    if (!$res->successful()) {
        return response()->json([
            'success' => false,
            'message' => 'Biteship reject request.',
            'biteship_status' => $res->status(),
            'biteship' => $res->json(),
            'sent_payload' => $payload,
        ], $res->status());
    }

    $body = $res->json();

    $biteshipOrderId = data_get($body, 'id') ?? data_get($body, 'data.id');
    $trackingNumber  = data_get($body, 'courier.tracking_id')
        ?? data_get($body, 'tracking_number')
        ?? data_get($body, 'data.tracking_number');

    $update = [];
    if ($biteshipOrderId) $update['biteship_order_id'] = $biteshipOrderId;
    if ($trackingNumber)  $update['shipping_tracking_number'] = $trackingNumber;

    if (!empty($update)) {
        $order->update($update);
    }

    return response()->json([
        'success' => true,
        'message' => 'Biteship order created.',
        'biteship' => $body,
        'saved' => $update,
    ], 200);
}


    /**
     * GET /api/shipping/orders/{orderId}
     */
    public function track($orderId)
    {
        $res = $this->biteship->track($orderId);
        return response()->json($res->json(), $res->status());
    }
}
