<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\UserAddress;
use App\Models\OtpVerification;
use App\Models\PasswordReset;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use App\Mail\ResetPasswordOtpMail;

class AuthCustomerController extends Controller
{
    /* ============ COMPLETE PROFILE & REGISTER ============ */
    public function store(Request $request)
        {
            $data = $request->validate([
                'label'            => 'nullable|string|max:50',
                'recipient_name'   => 'required|string|max:100',
                'phone_number'     => 'required|string|max:20',
                'address_full'     => 'required|string',
                'postal_code'      => 'required|string|max:10',

                // ✅ TAMBAHKAN INI
                'province_id'      => 'nullable|string|max:50',
                'city_id'          => 'nullable|string|max:50',
                'district_id'      => 'nullable|string|max:50',
                'biteship_area_id' => 'nullable|string|max:100',
                'is_primary'       => 'boolean',
            ]);

            $data['user_id'] = $request->user()->id;

            return UserAddress::create($data);
        }

    
    public function completeProfile(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'name' => 'required|string|max:100',
            'phone' => 'required|string|max:20',
            'password' => 'required|min:6',
            'address_full' => 'required|string',
            'postal_code' => 'required|string|max:10',
            'province_id'    => 'nullable|string',
            'city_id'        => 'nullable|string',
            'district_id'    => 'nullable|string',
            'biteship_area_id' => 'nullable|string',
            'is_primary'     => 'boolean'
        ]);

        $otp = OtpVerification::where('email', $request->email)->first();
        if (!$otp) {
            return response()->json([
                'message' => 'OTP belum diverifikasi'
            ], 403);
        }

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'password' => Hash::make($request->password),
            'role' => 'customer',
            'is_active' => 1
        ]);

        UserAddress::create([
            'user_id' => $user->id,
            'label' => 'Rumah',
            'recipient_name' => $request->name,
            'phone_number' => $request->phone,
            'address_full' => $request->address_full,
            'postal_code' => $request->postal_code,
            'is_primary' => 1
        ]);

        $otp->delete();

        $token = $user->createToken('customer-token')->plainTextToken;

        return response()->json([
            'message' => 'Registrasi berhasil',
            'token' => $token,
            'user' => $user
        ]);
    }

    /* ================= LOGIN ================= */
    public function loginCustomer(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required'
        ]);

        $user = User::where('email', $request->email)
            ->where('role', 'customer')
            ->where('is_active', 1)
            ->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json([
                'message' => 'Email atau password salah'
            ], 401);
        }

        $token = $user->createToken('customer-token')->plainTextToken;

        return response()->json([
            'message' => 'Login berhasil',
            'token' => $token,
            'user' => $user
        ]);
    }

    /* ================= LOGIN WITH OTP ================= */
    public function loginWithOtp(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'otp' => 'required',
            'password' => 'required'
        ]);

        $otp = OtpVerification::where('email', $request->email)
            ->where('otp_code', $request->otp)
            ->where('expires_at', '>', now())
            ->first();

        if (!$otp) {
            return response()->json([
                'message' => 'OTP tidak valid atau kadaluarsa'
            ], 422);
        }

        $user = User::where('email', $request->email)
            ->where('role', 'customer')
            ->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json([
                'message' => 'Password salah'
            ], 401);
        }

        $token = $user->createToken('customer-token')->plainTextToken;
        $otp->delete();

        return response()->json([
            'message' => 'Login berhasil',
            'token' => $token,
            'user' => $user
        ]);
    }

    /* ================= CHECK PASSWORD ================= */
    public function checkPassword(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required'
        ]);

        $user = User::where('email', $request->email)
            ->where('role', 'customer')
            ->where('is_active', 1)
            ->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json([
                'message' => 'Password salah'
            ], 401);
        }

        return response()->json([
            'message' => 'Password benar'
        ]);
    }

    /* ================= SEND RESET PASSWORD OTP ================= */
    public function sendResetPasswordOtp(Request $request)
    {
        $request->validate([
            'email' => 'required|email'
        ]);

        $user = User::where('email', $request->email)
            ->where('role', 'customer')
            ->first();

        if (!$user) {
            return response()->json([
                'message' => 'Email tidak terdaftar'
            ], 404);
        }

        $otp = rand(100000, 999999);

        PasswordReset::where('email', $request->email)->delete();

        PasswordReset::create([
            'email' => $request->email,
            'token' => $otp,
            'expires_at' => now()->addMinutes(5)
        ]);

        // ✅ KIRIM OTP KE EMAIL
        Mail::to($request->email)->send(
            new ResetPasswordOtpMail($otp, $request->email)
        );

        return response()->json([
            'message' => 'OTP reset password berhasil dikirim ke email'
        ]);
    }

    /* ================= RESET PASSWORD ================= */
    public function resetPassword(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'otp' => 'required',
            'password' => 'required|min:6|confirmed'
        ]);

        $reset = PasswordReset::where('email', $request->email)
            ->where('token', $request->otp)
            ->where('expires_at', '>', now())
            ->first();

        if (!$reset) {
            return response()->json([
                'message' => 'OTP tidak valid atau kadaluarsa'
            ], 422);
        }

        $user = User::where('email', $request->email)
            ->where('role', 'customer')
            ->first();

        if (!$user) {
            return response()->json([
                'message' => 'User tidak ditemukan'
            ], 404);
        }

        $user->update([
            'password' => Hash::make($request->password)
        ]);

        $reset->delete();

        return response()->json([
            'message' => 'Password berhasil direset'
        ]);
    }

    /* ================= LOGOUT ================= */
    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'message' => 'Logout berhasil'
        ]);
    }
}
