<?php

namespace App\Services;

use App\Models\Order;

class InvoiceService
{
    public static function statusLabel(?string $status): string
    {
        $s = strtolower($status ?? '');
        return match ($s) {
            'pending'     => 'Menunggu',
            'processing'  => 'Diproses',
            'shipped'     => 'Dikirim',
            'delivered'   => 'Selesai',
            'cancelled'   => 'Dibatalkan',
            'returned'    => 'Retur',
            default       => strtoupper($status ?? '-'),
        };
    }

    public static function buildEmailData(Order $order): array
    {
        $order->loadMissing(['user', 'items']);

        $items = $order->items->map(fn ($it) => [
            'product_name' => (string) $it->product_name,
            'variant_name' => (string) ($it->variant_name ?? ''),
            'qty'          => (int) $it->quantity,
            'price'        => (float) $it->price,
            'subtotal'     => (float) $it->price * (int) $it->quantity,
        ])->values()->all();

        $shipping = (float) ($order->shipping_cost ?? 0);
        $total    = (float) ($order->total_amount ?? 0);
        $subtotal = max(0, $total - $shipping);

        return [
            'order_id'            => $order->id,
            'invoice_number'      => $order->invoice_number,
            'order_status'        => $order->order_status,
            'order_status_label'  => self::statusLabel($order->order_status),

            'customer_name'       => $order->delivery_recipient_name,
            'customer_phone'      => $order->delivery_phone,
            'delivery_address'    => $order->delivery_address,
            'postal_code'         => $order->delivery_postal_code,

            'shipping_courier'    => $order->shipping_courier,
            'shipping_service'    => $order->shipping_service,
            'tracking_number'     => $order->shipping_tracking_number,

            'subtotal'            => $subtotal,
            'shipping_cost'       => $shipping,
            'total'               => $total,

            'items'               => $items,
            'created_at'          => optional($order->created_at)->format('d M Y H:i'),
            'paid_at'             => optional($order->paid_at)->format('d M Y H:i'),

            // ✅ penting untuk frontend disable button
            'invoice_emailed_at'  => optional($order->invoice_emailed_at)->format('d M Y H:i'),
        ];
    }
}
