<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\UserAddress;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class ShippingController extends Controller
{
    public function store(Request $request)
    {
        $data = $request->validate([
            // shipping (WAJIB)
            'shipping_courier' => ['required', 'string'],
            'shipping_service' => ['required', 'string'],
            'shipping_cost' => ['required', 'numeric', 'min:0'],

            // OPTIONAL: kalau FE kirim address_id (alamat yang dipilih)
            'address_id' => ['nullable', 'integer', 'exists:user_addresses,id'],

            // OPTIONAL: kalau FE kirim postal code (dipakai untuk biteship)
            'delivery_postal_code' => ['nullable', 'string', 'min:5'],

            // items
            'items' => ['required', 'array', 'min:1'],
            'items.*.product_id' => ['required', 'integer'],
            'items.*.variant_id' => ['required', 'integer'],
            'items.*.product_name' => ['required', 'string'],
            'items.*.variant_name' => ['required', 'string'],
            'items.*.quantity' => ['required', 'integer', 'min:1'],
            'items.*.price' => ['required', 'numeric', 'min:0'],
            'items.*.hpp' => ['required', 'numeric', 'min:0'],

            // OPTIONAL: kalau kamu mau simpan weight untuk create order biteship dari local
            'items.*.weight' => ['nullable', 'numeric', 'min:1'],
        ]);

        $user = $request->user();

        // 1) Ambil alamat yang dipilih (kalau ada), kalau tidak ambil primary
        $addressQuery = UserAddress::query()->where('user_id', $user->id);

        if (!empty($data['address_id'])) {
            $addressQuery->where('id', (int) $data['address_id']);
        } else {
            $addressQuery->where('is_primary', 1);
        }

        $address = $addressQuery->first();

        if (!$address) {
            return response()->json([
                'success' => false,
                'message' => 'Alamat belum tersedia / belum dipilih.',
            ], 422);
        }

        // 2) Pastikan postal code ada (untuk Biteship)
        $postal = $data['delivery_postal_code'] ?? $address->postal_code;
        $postalDigits = preg_replace('/\D/', '', (string) $postal);

        if (!$postalDigits || strlen($postalDigits) < 5) {
            return response()->json([
                'success' => false,
                'message' => 'Kode pos tujuan belum ada. Lengkapi kode pos alamat.',
            ], 422);
        }

        // 3) NORMALISASI courier/service (INI KUNCI BIAR SESUAI BITESHIP)
        $courierCode = $this->normalizeCourierCode($data['shipping_courier']);
        $serviceCode = $this->normalizeServiceCode($data['shipping_service']);

        if (!$courierCode || !$serviceCode) {
            return response()->json([
                'success' => false,
                'message' => 'Kurir / service tidak valid. Pastikan yang dikirim adalah courier_code dan courier_service_code dari Biteship.',
                'debug' => [
                    'shipping_courier_in' => $data['shipping_courier'],
                    'shipping_service_in' => $data['shipping_service'],
                    'shipping_courier_normalized' => $courierCode,
                    'shipping_service_normalized' => $serviceCode,
                ],
            ], 422);
        }

        // 4) Hitung subtotal & total HPP
        $subtotal = collect($data['items'])->sum(function ($i) {
            return (float) $i['price'] * (int) $i['quantity'];
        });

        $totalHpp = collect($data['items'])->sum(function ($i) {
            return (float) $i['hpp'] * (int) $i['quantity'];
        });

        $shippingCost = (float) $data['shipping_cost'];
        $totalAmount  = $subtotal + $shippingCost;

        // 5) Simpan Order + Items dengan transaction
        try {
            $order = DB::transaction(function () use (
                $user,
                $address,
                $postalDigits,
                $courierCode,
                $serviceCode,
                $data,
                $totalHpp,
                $shippingCost,
                $totalAmount
            ) {
                $invoice = $this->generateUniqueInvoice();

                $order = Order::create([
                    'user_id' => $user->id,

                    // alamat (ambil dari address)
                    'delivery_address' => $address->address_full,
                    'delivery_postal_code' => (string) $postalDigits,
                    'delivery_recipient_name' => $address->recipient_name,
                    'delivery_phone' => $address->phone_number,

                    // meta order
                    'invoice_number' => $invoice,
                    'transaction_source' => 'online',

                    // status default
                    'order_status' => 'pending',
                    'payment_status' => 'pending',

                    // totals
                    'total_amount' => $totalAmount,
                    'total_hpp' => $totalHpp,

                    // shipping detail (SIMPAN YANG SUDAH DINORMALISASI)
                    'shipping_cost' => $shippingCost,
                    'shipping_courier' => $courierCode, // contoh: jne
                    'shipping_service' => $serviceCode, // contoh: reg
                ]);

                foreach ($data['items'] as $item) {
                    OrderItem::create([
                        'order_id' => $order->id,
                        'product_id' => (int) $item['product_id'],
                        'variant_id' => (int) $item['variant_id'],
                        'product_name' => (string) $item['product_name'],
                        'variant_name' => (string) $item['variant_name'],
                        'quantity' => (int) $item['quantity'],
                        'price' => (float) $item['price'],
                        'hpp' => (float) $item['hpp'],
                        // 'weight' => isset($item['weight']) ? (int) $item['weight'] : null,
                    ]);
                }

                return $order;
            });

            return response()->json([
                'success' => true,
                'message' => 'Order berhasil dibuat',
                'order' => $order->load('items'),
            ], 201);

        } catch (\Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Checkout gagal',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    private function normalizeCourierCode(string $value): string
    {
        // Jaga-jaga kalau ada input seperti "JNE" / "jne " / "JNE REG"
        $v = strtolower(trim($value));
        $v = preg_replace('/\s+/', ' ', $v);

        // kalau ada yang nyelip "jne reg", ambil token pertama
        $parts = explode(' ', $v);
        $v = $parts[0] ?? $v;

        // mapping umum (optional)
        $map = [
            'j&t' => 'jnt',
            'jnt' => 'jnt',
        ];

        return $map[$v] ?? $v;
    }

    private function normalizeServiceCode(string $value): string
    {
        $v = strtolower(trim($value));
        $v = preg_replace('/\s+/', ' ', $v);

        // mapping umum (karena sering FE simpan "reguler")
        $map = [
            'reguler' => 'reg',
            'regular' => 'reg',
            'standard' => 'reg',
        ];

        $v = $map[$v] ?? $v;

        // kalau ada "reg - something", ambil token pertama
        $parts = preg_split('/[^a-z0-9]+/i', $v);
        $v = $parts[0] ?? $v;

        return $v;
    }

    private function generateUniqueInvoice(): string
    {
        do {
            $invoice = 'INV-' . now()->format('Ymd') . '-' . strtoupper(Str::random(6));
        } while (Order::where('invoice_number', $invoice)->exists());

        return $invoice;
    }
}
