<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class ProfileController extends Controller
{
    /* =========================
       HELPERS
    ========================= */

    private function avatarPublicUrl(string $filename): string
    {
        // Karena file benar-benar ada di public/storage/avatars
        // maka URL publiknya adalah /storage/avatars/<filename>
        return asset('storage/avatars/' . $filename);
    }

    private function buildAvatarUrl($user): ?string
    {
        // jika pakai kolom avatar berisi filename: "avatar_1_123.png"
        if (!empty($user->avatar)) {
            // cek file memang ada di public/storage/avatars
            $fullPath = public_path('storage/avatars/' . $user->avatar);
            if (file_exists($fullPath)) {
                return $this->avatarPublicUrl($user->avatar);
            }

            // fallback tetap balikin url (biar frontend tetap coba render)
            return $this->avatarPublicUrl($user->avatar);
        }

        // jika ada kolom avatar_url legacy (mis. sudah full url atau path)
        if (!empty($user->avatar_url)) {
            $raw = (string) $user->avatar_url;

            // sudah absolute
            if (preg_match('/^https?:\/\//i', $raw)) {
                return $raw;
            }

            // pastikan ada leading slash
            $path = str_starts_with($raw, '/') ? $raw : '/' . $raw;

            return asset($path);
        }

        return null;
    }

    private function attachAvatarUrl($user)
    {
        $user->avatar_url = $this->buildAvatarUrl($user);
        return $user;
    }

    /* =========================
       PROFILE
    ========================= */
    public function show(Request $request)
    {
        $user = $request->user();

        // ✅ pastikan avatar_url selalu siap dipakai frontend
        $this->attachAvatarUrl($user);

        return response()->json($user);
    }

    public function update(Request $request)
{
    $data = $request->validate([
        'name'        => 'required|string|max:100',
        'phone'       => 'nullable|string|max:20',
        'birth_date'  => 'nullable|date',
        // DB enum: male/female, tapi terima juga input lama agar kompatibel
        'gender'      => 'nullable|in:male,female,Laki-laki,Perempuan',
    ]);

    // Normalisasi gender ke nilai DB
    if (isset($data['gender'])) {
        if ($data['gender'] === 'Laki-laki') $data['gender'] = 'male';
        if ($data['gender'] === 'Perempuan') $data['gender'] = 'female';
        if ($data['gender'] === '') $data['gender'] = null;
    }

    $user = $request->user();
    $user->update($data);

    $this->attachAvatarUrl($user);

    return response()->json([
        'message' => 'Profil berhasil diperbarui',
        'user'    => $user,
    ]);
}


    /* =========================
       AVATAR
    ========================= */
    public function updateAvatar(Request $request)
    {
        $request->validate([
            'avatar' => 'required|image|mimes:jpg,jpeg,png|max:10240', // 10MB
        ]);

        $user = $request->user();

        // ✅ Pastikan folder ada: public/storage/avatars
        $dir = public_path('storage/avatars');
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }

        // ✅ hapus avatar lama kalau ada (di public/storage/avatars)
        if (!empty($user->avatar)) {
            $oldPath = public_path('storage/avatars/' . $user->avatar);
            if (file_exists($oldPath)) {
                @unlink($oldPath);
            }
        }

        $file = $request->file('avatar');
        $filename = 'avatar_' . $user->id . '_' . time() . '.' . $file->getClientOriginalExtension();

        // ✅ SIMPAN LANGSUNG ke public/storage/avatars
        $file->move($dir, $filename);

        // ✅ update db
        $user->update([
            'avatar' => $filename,
        ]);

        // ✅ bikin url publik yang benar
        $avatarUrl = $this->avatarPublicUrl($filename);

        return response()->json([
            'message'    => 'Foto profil berhasil diubah',
            'avatar'     => $filename,
            'avatar_url' => $avatarUrl,
            'user'       => $this->attachAvatarUrl($user->fresh()),
        ]);
    }

    /* =========================
       PASSWORD
    ========================= */
    public function updatePassword(Request $request)
    {
        $request->validate([
            'password' => 'required|min:8|confirmed',
        ]);

        $request->user()->update([
            'password' => Hash::make($request->password),
        ]);

        return response()->json([
            'message' => 'Password berhasil diubah',
        ]);
    }
}
