<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductVariant;
use App\Models\ProductImage;
use Illuminate\Http\Request;

class ProductVariantController extends Controller
{
    /**
     * ✅ TAMBAHAN (TIDAK MENGGANGGU YANG LAIN):
     * GET /api/variants?ids=1,2,3
     * Dipakai Checkout.jsx untuk sync items by variant ids
     */
    public function byIds(Request $request)
    {
        $idsParam = (string) $request->query('ids', '');
        $ids = array_values(array_filter(array_map('intval', explode(',', $idsParam))));

        if (empty($ids)) {
            return response()->json([]);
        }

        // ambil variants + images (sesuai kebutuhan FE kamu)
        $variants = ProductVariant::query()
            ->with('images')
            ->whereIn('id', $ids)
            ->get();

        return response()->json($variants);
    }

    public function index(Product $product)
    {
        // ✅ ikutkan images biar FE bisa nampilin gambar varian
        return response()->json(
            $product->variants()->with('images')->latest('id')->get()
        );
    }

    public function store(Request $request, Product $product)
    {
        $data = $request->validate([
            'variant_name' => 'nullable|string|max:200',
            'name'         => 'nullable|string|max:200',
            'sku'          => 'nullable|string|max:100',
            'price'        => 'nullable|numeric|min:0',
            'hpp'          => 'nullable|numeric|min:0',
            'stock'        => 'nullable|integer|min:0',
            'image_url'    => 'nullable|string|max:500',
        ]);

        // ✅ ambil image_url, jangan dimasukkan ke product_variants
        $imageUrl = $data['image_url'] ?? null;
        unset($data['image_url']);

        // mapping name
        $name = $data['name']
            ?? $data['variant_name']
            ?? $product->name;

        $variant = $product->variants()->create([
            'name'  => $name,
            'sku'   => $data['sku'] ?? null, // ✅ SKU untuk variants
            'price' => $data['price'] ?? 0,
            'hpp'   => $data['hpp'] ?? 0, // ✅ default 0
            'stock' => $data['stock'] ?? 0,
        ]);

        // ✅ simpan gambar varian ke product_images
        if (!empty($imageUrl)) {
            ProductImage::create([
                'product_id' => $product->id,
                'variant_id' => $variant->id,
                'image_url'  => $imageUrl,
                'is_cover'   => 1,
            ]);
        }

        return response()->json($variant->load('images'), 201);
    }

    public function update(Request $request, Product $product, ProductVariant $variant)
    {
        if ($variant->product_id !== $product->id) {
            return response()->json(['message' => 'Variant tidak milik product ini.'], 404);
        }

        $data = $request->validate([
            'variant_name' => 'nullable|string|max:200',
            'name'         => 'nullable|string|max:200',
            'sku'          => 'nullable|string|max:100',
            'price'        => 'nullable|numeric|min:0',
            'hpp'          => 'nullable|numeric|min:0',
            'stock'        => 'nullable|integer|min:0',
            'image_url'    => 'nullable|string|max:500',
        ]);

        // mapping variant_name -> name (kalau name gak dikirim)
        if (array_key_exists('variant_name', $data) && !array_key_exists('name', $data)) {
            $data['name'] = $data['variant_name'];
        }
        unset($data['variant_name']);

        // ✅ ambil image_url, jangan dimasukkan ke update variants
        $imageUrl = $data['image_url'] ?? null;
        unset($data['image_url']);

        // ✅ update hanya kolom variants yang memang ada
        $variant->update([
            'name'  => $data['name'] ?? $variant->name,
            'sku'   => $data['sku'] ?? $variant->sku, // ✅ SKU untuk variants
            'price' => array_key_exists('price', $data) ? ($data['price'] ?? 0) : $variant->price,
            'hpp'   => array_key_exists('hpp', $data) ? ($data['hpp'] ?? 0) : $variant->hpp,
            'stock' => array_key_exists('stock', $data) ? ($data['stock'] ?? 0) : $variant->stock,
        ]);

        // ✅ upsert cover image varian
        if (!empty($imageUrl)) {
            ProductImage::updateOrCreate(
                ['variant_id' => $variant->id, 'is_cover' => 1],
                [
                    'product_id' => $product->id,
                    'image_url'  => $imageUrl,
                ]
            );
        }

        return response()->json($variant->load('images'));
    }

    public function destroy(Product $product, ProductVariant $variant)
    {
        if ($variant->product_id !== $product->id) {
            return response()->json(['message' => 'Variant tidak milik product ini.'], 404);
        }

        // ✅ hapus juga gambar yang terkait varian (opsional tapi bagus)
        ProductImage::where('variant_id', $variant->id)->delete();

        $variant->delete();
        return response()->json(['message' => 'Variant deleted']);
    }
}
