<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ProductImage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ProductImageController extends Controller
{
    public function index(Request $request)
    {
        $productId = $request->get('product_id');
        $variantId = $request->get('variant_id');
        $isCover   = $request->get('is_cover'); // 0/1
        $perPage   = $request->get('per_page'); // optional

        $q = ProductImage::query()->latest('id');

        if ($productId !== null && $productId !== '') {
            $q->where('product_id', (int) $productId);
        }

        // penting: bedakan null vs ada value
        if ($variantId !== null && $variantId !== '') {
            $q->where('variant_id', (int) $variantId);
        }

        if ($isCover !== null && $isCover !== '') {
            $q->where('is_cover', (int) $isCover);
        }

        // default: pagination kalau per_page ada, jika tidak return array data
        if (!empty($perPage)) {
            $perPage = max(1, min((int) $perPage, 200));
            return response()->json($q->paginate($perPage));
        }

        return response()->json([
            'data' => $q->get()
        ]);
    }

    public function show(ProductImage $productImage)
    {
        $productImage->load('product');
        return response()->json($productImage);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'product_id' => ['required', 'integer', 'exists:products,id'],
            'variant_id' => ['nullable', 'integer', 'exists:product_variants,id'],
            'image_url'  => ['required', 'string', 'max:500'],
            'is_cover'   => ['nullable', 'boolean'],
        ]);

        $validated['is_cover'] = (int) ($validated['is_cover'] ?? 0);

        return DB::transaction(function () use ($validated) {
            // ✅ jika set cover, matikan cover lain di scope yang sama
            if ($validated['is_cover'] === 1) {
                ProductImage::query()
                    ->where('product_id', $validated['product_id'])
                    ->when(
                        array_key_exists('variant_id', $validated) && $validated['variant_id'] !== null,
                        fn($qq) => $qq->where('variant_id', $validated['variant_id']),
                        fn($qq) => $qq->whereNull('variant_id')
                    )
                    ->update(['is_cover' => 0]);
            }

            $image = ProductImage::create($validated);
            return response()->json($image, 201);
        });
    }

    public function update(Request $request, ProductImage $productImage)
    {
        $validated = $request->validate([
            'image_url' => ['sometimes', 'string', 'max:500'],
            'is_cover'  => ['nullable', 'boolean'],
        ]);

        return DB::transaction(function () use ($validated, $productImage) {
            if (array_key_exists('is_cover', $validated)) {
                $validated['is_cover'] = (int) ($validated['is_cover'] ?? 0);

                // ✅ jika set cover, matikan cover lain di scope yang sama
                if ($validated['is_cover'] === 1) {
                    ProductImage::query()
                        ->where('product_id', $productImage->product_id)
                        ->when(
                            $productImage->variant_id !== null,
                            fn($qq) => $qq->where('variant_id', $productImage->variant_id),
                            fn($qq) => $qq->whereNull('variant_id')
                        )
                        ->where('id', '!=', $productImage->id)
                        ->update(['is_cover' => 0]);
                }
            }

            $productImage->update($validated);
            return response()->json($productImage);
        });
    }

    public function destroy(ProductImage $productImage)
    {
        $productImage->delete();
        return response()->json(['message' => 'Product image deleted']);
    }
}
