<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $q          = trim((string) ($request->get('q', '') ?: $request->get('search', '')));
        $categoryId = $request->get('category_id');
        $brand      = $request->get('brand');      // slug brand
        $perPage    = $request->get('per_page');   // optional

        $query = Product::query()
            ->with(['category', 'brand'])
            ->withCount('variants')
            ->withSum('variants as stock_total', 'stock')
            ->withMin('variants as price_min', 'price')
            ->withMax('variants as price_max', 'price')
            ->withMin('variants as hpp_min', 'hpp')
            ->withMax('variants as hpp_max', 'hpp')

            // ✅ rating dari product_reviews
            ->withAvg('reviews as rating_avg', 'rating')
            ->withCount('reviews as rating_count')

            // ✅ sold_count aman: hitung jumlah item order_items per product
            ->addSelect([
                'sold_count' => DB::table('order_items')
                    ->join('orders', 'orders.id', '=', 'order_items.order_id')
                    ->whereColumn('order_items.product_id', 'products.id')
                    ->whereIn('orders.order_status', ['processing', 'shipped', 'delivered'])
                    ->selectRaw('COUNT(*)')
            ])
            ->latest('id');

        /**
         * 🔍 SEARCH
         */
        if ($q !== '') {
            $query->where(function ($w) use ($q) {
                $w->where('name', 'like', "%{$q}%")
                    ->orWhere('sku', 'like', "%{$q}%"); // ✅ SKU produk (products.sku)
            });
        }

        /**
         * 📁 CATEGORY FILTER
         */
        if (!empty($categoryId)) {
            $query->where('category_id', $categoryId);
        }

        /**
         * 🏷 BRAND FILTER (by slug)
         */
        if (!empty($brand)) {
            $query->whereHas('brand', function ($b) use ($brand) {
                $b->where('slug', $brand);
            });
        }

        // 👉 ADMIN / LIST PAGE (PAGINATION)
        if (!empty($perPage)) {
            $perPage = max(1, min((int) $perPage, 100));
            return response()->json($query->paginate($perPage));
        }

        // 👉 CATEGORY PAGE / FRONTEND (AMBIL SEMUA)
        return response()->json([
            'data' => $query->get()
        ]);
    }

    public function show(Product $product)
    {
        $product->load([
            'category',
            'variants.images', // ✅ FIX: ikutkan images varian (product_images)
            'images',
            'reviews' => function ($q) {
                $q->with(['user'])
                    ->latest('id');
            },
        ]);

        $ratingAvg = (float) ($product->reviews()->avg('rating') ?? 0);
        $ratingCount = (int) ($product->reviews()->count() ?? 0);

        $soldCount = (int) DB::table('order_items')
            ->join('orders', 'orders.id', '=', 'order_items.order_id')
            ->where('order_items.product_id', $product->id)
            ->whereIn('orders.order_status', ['processing', 'shipped', 'delivered'])
            ->count();

        return response()->json([
            ...$product->toArray(),
            'rating_avg' => $ratingAvg,
            'rating_count' => $ratingCount,
            'sold_count' => $soldCount,
        ]);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'category_id'    => 'required|exists:categories,id',
            'brand_id'       => 'nullable|exists:brands,id',
            'name'           => 'required|string|max:200',
            'slug'           => 'nullable|string|max:255|unique:products,slug',
            'sku'            => 'nullable|string|max:50|unique:products,sku',
            'description'    => 'nullable|string',
            'thumbnail_url'  => 'nullable|string',
            'weight_grams'   => 'required|integer',

            // ✅ dimensi boleh null
            'length_cm'      => 'nullable|integer|min:0',
            'width_cm'       => 'nullable|integer|min:0',
            'height_cm'      => 'nullable|integer|min:0',

            'is_active'      => 'boolean',
        ]);

        // ✅ normalisasi: empty string -> null (aman kalau FE kadang ngirim "")
        foreach (['length_cm', 'width_cm', 'height_cm'] as $k) {
            if (array_key_exists($k, $data) && ($data[$k] === '' || $data[$k] === null)) {
                $data[$k] = null;
            }
        }

        // generate slug kalau kosong
        if (empty($data['slug'])) {
            $base = Str::slug($data['name']);
            $slug = $base ?: 'product';

            $i = 2;
            while (Product::where('slug', $slug)->exists()) {
                $slug = $base . '-' . $i;
                $i++;
            }
            $data['slug'] = $slug;
        }

        $data['is_active'] = $data['is_active'] ?? 1;
        $data['brand_id']  = $data['brand_id'] ?? null;

        $product = Product::create($data);
        return response()->json($product, 201);
    }

    public function update(Request $request, Product $product)
    {
        $data = $request->validate([
            'category_id'   => 'sometimes|exists:categories,id',
            'name'          => 'sometimes|string|max:200',
            'brand_id'      => 'sometimes|nullable|exists:brands,id',
            'slug'          => 'nullable|string|max:255|unique:products,slug,' . $product->id,
            'sku'           => 'nullable|string|max:50|unique:products,sku,' . $product->id,
            'description'   => 'nullable|string',
            'thumbnail_url' => 'nullable|string',
            'weight_grams'  => 'sometimes|integer',

            // ✅ INI yang dibenerin: update juga boleh null
            'length_cm'     => 'sometimes|nullable|integer|min:0',
            'width_cm'      => 'sometimes|nullable|integer|min:0',
            'height_cm'     => 'sometimes|nullable|integer|min:0',

            'is_active'     => 'boolean',
        ]);

        // ✅ normalisasi: kalau dikirim kosong/null, simpan null (bukan 0)
        foreach (['length_cm', 'width_cm', 'height_cm'] as $k) {
            if (array_key_exists($k, $data) && ($data[$k] === '' || $data[$k] === null)) {
                $data[$k] = null;
            }
        }

        // ✅ update hanya field yang boleh (tanpa ganggu fungsi lain)
        $product->update($data);

        return response()->json($product);
    }

    public function destroy(Product $product)
    {
        $product->delete();
        return response()->json(['message' => 'Product deleted']);
    }
}
