<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\PosSession;
use Illuminate\Http\Request;

class PosSessionController extends Controller
{
    public function index()
    {
        $data = PosSession::with(['branch', 'user'])
            ->latest('id')
            ->paginate(10);

        return response()->json($data);
    }

    public function show(PosSession $posSession)
    {
        $posSession->load(['branch', 'user']);
        return response()->json($posSession);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'branch_id'     => ['required', 'integer', 'exists:branches,id'],
            'user_id'       => ['required', 'integer', 'exists:users,id'],
            'opening_cash'  => ['required', 'numeric', 'min:0'],
            'closing_cash'  => ['nullable', 'numeric', 'min:0'],
            'total_sales'   => ['nullable', 'numeric', 'min:0'],
            'actual_cash'   => ['nullable', 'numeric', 'min:0'],
            'start_time'    => ['required', 'date'],
            'end_time'      => ['nullable', 'date'],
            'status'        => ['required', 'in:open,closed'],
            'note'          => ['nullable', 'string'],
        ]);

        $session = PosSession::create($validated);

        return response()->json($session, 201);
    }

    public function update(Request $request, PosSession $posSession)
    {
        $validated = $request->validate([
            'opening_cash'  => ['sometimes', 'numeric', 'min:0'],
            'closing_cash'  => ['nullable', 'numeric', 'min:0'],
            'total_sales'   => ['nullable', 'numeric', 'min:0'],
            'actual_cash'   => ['nullable', 'numeric', 'min:0'],
            'start_time'    => ['sometimes', 'date'],
            'end_time'      => ['nullable', 'date'],
            'status'        => ['sometimes', 'in:open,closed'],
            'note'          => ['nullable', 'string'],
        ]);

        $posSession->update($validated);

        return response()->json($posSession);
    }

    public function destroy(PosSession $posSession)
    {
        $posSession->delete();
        return response()->json(['message' => 'POS session deleted']);
    }
}