<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Midtrans\Config;
use Midtrans\Snap;
use Midtrans\Transaction;
use App\Models\Order;

class PaymentController extends Controller
{
    private function initMidtrans(): void
    {
        Config::$serverKey    = config('midtrans.server_key');
        Config::$clientKey    = config('midtrans.client_key'); // optional
        Config::$isProduction = (bool) config('midtrans.is_production');
        Config::$isSanitized  = (bool) config('midtrans.is_sanitized', true);
        Config::$is3ds        = (bool) config('midtrans.is_3ds', true);
    }

    public function createMidtrans(Request $request)
    {
        $request->validate([
            'order_id' => 'required|exists:orders,id',
        ]);

        $order = Order::findOrFail($request->order_id);

        if ($order->payment_status === 'paid') {
            return response()->json(['message' => 'Order sudah dibayar'], 400);
        }

        $this->initMidtrans();

        // base order_id untuk midtrans
        $baseMidtransOrderId = (string) $order->invoice_number;

        // builder params
        $makeParams = function (string $midtransOrderId) use ($order) {
            return [
                'transaction_details' => [
                    'order_id'     => $midtransOrderId,
                    'gross_amount' => (int) $order->total_amount,
                ],
                'customer_details' => [
                    'first_name' => $order->delivery_recipient_name ?? 'Customer',
                    'phone'      => $order->delivery_phone ?? '-',
                ],
            ];
        };

        try {
            // 1) coba pertama pakai invoice_number (ideal)
            $params = $makeParams($baseMidtransOrderId);
            $snapToken = Snap::getSnapToken($params);

            // simpan midtrans_order_id dan payment_method
            $order->update([
                'payment_method' => 'midtrans',
                'midtrans_order_id' => $baseMidtransOrderId,
            ]);

            return response()->json([
                'snap_token' => $snapToken,
                'midtrans_order_id' => $baseMidtransOrderId,
            ]);
        } catch (\Throwable $e) {
            $msg = $e->getMessage();

            Log::warning('MIDTRANS SNAP TOKEN FIRST TRY FAILED', [
                'order_id' => $order->id,
                'invoice_number' => $order->invoice_number,
                'midtrans_order_id' => $baseMidtransOrderId,
                'error' => $msg,
            ]);

            // 2) kalau error karena duplicate order_id -> retry pakai suffix unik
            if ($this->isDuplicateOrderIdError($msg)) {
                $retryMidtransOrderId = $baseMidtransOrderId . '-R' . now()->format('YmdHis');

                try {
                    $params2 = $makeParams($retryMidtransOrderId);
                    $snapToken2 = Snap::getSnapToken($params2);

                    $order->update([
                        'payment_method' => 'midtrans',
                        'midtrans_order_id' => $retryMidtransOrderId,
                    ]);

                    return response()->json([
                        'snap_token' => $snapToken2,
                        'midtrans_order_id' => $retryMidtransOrderId,
                        'note' => 'retry_with_new_midtrans_order_id',
                    ]);
                } catch (\Throwable $e2) {
                    Log::error('MIDTRANS SNAP TOKEN RETRY FAILED', [
                        'order_id' => $order->id,
                        'invoice_number' => $order->invoice_number,
                        'midtrans_order_id' => $retryMidtransOrderId,
                        'error' => $e2->getMessage(),
                    ]);

                    return response()->json([
                        'message' => 'Gagal membuat Snap Token (retry)',
                    ], 500);
                }
            }

            Log::error('MIDTRANS SNAP TOKEN ERROR', [
                'order_id' => $order->id,
                'invoice_number' => $order->invoice_number,
                'midtrans_order_id' => $baseMidtransOrderId,
                'error' => $msg,
            ]);

            return response()->json([
                'message' => 'Gagal membuat Snap Token',
            ], 500);
        }
    }

    /**
     * GET /api/orders/{order}/status
     * Catatan penting:
     * - Midtrans "pending" itu artinya transaksi ada tapi belum dibayar.
     * - FE kamu saat ini menganggap "pending" = arahkan ke /orders.
     * - Jadi field `status` kita mapping agar FE tidak mengunci user.
     * - Status asli Midtrans tetap dikirim di `midtrans_status`.
     */
    public function orderStatus(Request $request, Order $order)
    {
        try {
            $this->initMidtrans();

            $base = (string) ($order->invoice_number ?? '');
            $stored = trim((string) ($order->midtrans_order_id ?? ''));

            if ($base === '' && $stored === '') {
                return response()->json([
                    'success' => false,
                    'message' => 'invoice_number & midtrans_order_id kosong, tidak bisa cek status Midtrans.',
                    'order_id' => $order->id,
                    'payment_status' => $order->payment_status ?? null,
                ], 200);
            }

            $candidates = [];

            $qTry = trim((string) $request->query('midtrans_order_id', ''));
            if ($qTry !== '') $candidates[] = $qTry;

            if ($stored !== '') $candidates[] = $stored;
            if ($base !== '') $candidates[] = $base;

            $candidates = array_values(array_unique(array_filter($candidates)));

            foreach ($candidates as $midtransOrderId) {
                try {
                    $midtrans = Transaction::status($midtransOrderId);

                    // ambil status asli dari Midtrans
                    $txStatus = null;
                    if (is_object($midtrans) && isset($midtrans->transaction_status)) {
                        $txStatus = $midtrans->transaction_status;
                    } elseif (is_array($midtrans) && isset($midtrans['transaction_status'])) {
                        $txStatus = $midtrans['transaction_status'];
                    } elseif (is_object($midtrans) && isset($midtrans->status)) {
                        $txStatus = $midtrans->status;
                    } elseif (is_array($midtrans) && isset($midtrans['status'])) {
                        $txStatus = $midtrans['status'];
                    }

                    // ✅ mapping untuk FE agar "pending" tidak menghalangi bayar
                    // - pending (Midtrans) => unpaid (FE)
                    // - settlement/capture => paid
                    // - expire => expired
                    // - cancel/deny => failed
                    $statusForFE = $txStatus;

                    if (is_string($txStatus)) {
                        $s = strtolower($txStatus);

                        if ($s === 'pending') {
                            $statusForFE = 'unpaid'; // <-- kunci fix
                        } elseif ($s === 'settlement' || $s === 'capture') {
                            $statusForFE = 'paid';
                        } elseif ($s === 'expire') {
                            $statusForFE = 'expired';
                        } elseif ($s === 'cancel' || $s === 'deny') {
                            $statusForFE = 'failed';
                        }
                    }

                    return response()->json([
                        'success' => true,
                        'source' => 'midtrans',
                        'midtrans_order_id' => $midtransOrderId,
                        'order_id' => $order->id,
                        'invoice_number' => $order->invoice_number,
                        'payment_status' => $order->payment_status ?? null,

                        // ✅ FE baca ini (sekarang tidak akan "pending" lagi)
                        'status' => $statusForFE,

                        // ✅ status asli midtrans untuk debug / kebutuhan lain
                        'midtrans_status' => $txStatus,

                        'midtrans' => $midtrans,
                    ]);
                } catch (\Throwable $eTry) {
                    $m = $eTry->getMessage();
                    $isNotFound = str_contains($m, "404") && str_contains($m, "Transaction doesn't exist");

                    if (!$isNotFound) {
                        Log::warning('MIDTRANS STATUS TRY FAILED', [
                            'order_id' => $order->id,
                            'try' => $midtransOrderId,
                            'error' => $m,
                        ]);
                    }
                }
            }

            Log::warning('MIDTRANS STATUS FAILED (all candidates)', [
                'order_id' => $order->id,
                'candidates' => $candidates,
            ]);

            // fallback local: untuk FE juga jangan kirim "pending" supaya tidak memblok bayar
            $localStatusForFE = $order->payment_status ?? null;
            if (is_string($localStatusForFE) && strtolower($localStatusForFE) === 'pending') {
                $localStatusForFE = 'unpaid';
            }

            return response()->json([
                'success' => true,
                'source' => 'local',
                'order_id' => $order->id,
                'invoice_number' => $order->invoice_number,
                'payment_status' => $order->payment_status ?? null,
                'order_status' => $order->order_status ?? null,

                // ✅ FE baca ini
                'status' => $localStatusForFE,

                'message' => 'Midtrans transaksi tidak ditemukan / gagal diakses, mengembalikan status lokal.',
            ], 200);
        } catch (\Throwable $e) {
            Log::error('orderStatus fatal error', [
                'order_id' => $order->id ?? null,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Internal Server Error (orderStatus)',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function midtransWebhook(Request $request)
    {
        $payload = $request->all();
        Log::info('MIDTRANS WEBHOOK RAW', $payload);

        if (
            empty($payload['order_id']) ||
            empty($payload['status_code']) ||
            empty($payload['gross_amount']) ||
            empty($payload['signature_key'])
        ) {
            return response()->json(['message' => 'Invalid payload'], 400);
        }

        $serverKey = config('midtrans.server_key');

        $expectedSignature = hash(
            'sha512',
            $payload['order_id'] . $payload['status_code'] . $payload['gross_amount'] . $serverKey
        );

        if (!hash_equals($expectedSignature, $payload['signature_key'])) {
            Log::warning('MIDTRANS SIGNATURE INVALID', [
                'expected' => $expectedSignature,
                'got' => $payload['signature_key'],
            ]);
            return response()->json(['message' => 'Invalid signature'], 403);
        }

        $midtransOrderId = (string) $payload['order_id'];
        $invoiceBase = $this->extractInvoiceBase($midtransOrderId);

        $order = Order::where('invoice_number', $invoiceBase)->first();
        if (!$order) {
            return response()->json(['message' => 'Order not found'], 404);
        }

        $order->update([
            'payment_method' => 'midtrans',
            'midtrans_order_id' => $midtransOrderId,
        ]);

        $transactionStatus = $payload['transaction_status'] ?? null;
        $fraudStatus = $payload['fraud_status'] ?? null;

        if (in_array($transactionStatus, ['settlement', 'capture'], true)) {
            if ($transactionStatus === 'capture' && $fraudStatus === 'challenge') {
                $order->update(['payment_status' => 'pending']);
            } else {
                $order->update([
                    'payment_status' => 'paid',
                    'paid_at' => now(),
                ]);
            }
        } elseif ($transactionStatus === 'pending') {
            $order->update(['payment_status' => 'pending']);
        } elseif ($transactionStatus === 'expire') {
            $order->update(['payment_status' => 'expired']);
        } elseif (in_array($transactionStatus, ['cancel', 'deny'], true)) {
            $order->update(['payment_status' => 'failed']);
        }

        return response()->json(['success' => true]);
    }

    private function isDuplicateOrderIdError(string $message): bool
    {
        return str_contains($message, 'transaction_details.order_id has already been taken');
    }

    private function extractInvoiceBase(string $midtransOrderId): string
    {
        return preg_replace('/-R\d{14}$/', '', $midtransOrderId) ?: $midtransOrderId;
    }
}
