<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Mail\InvoiceMail;
use App\Models\Notification;
use App\Models\Order;
use App\Services\InvoiceService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;

class OrderInvoiceController extends Controller
{
    public function invoice(Request $request, Order $order)
    {
        $user = $request->user();

        if ((int)$order->user_id !== (int)$user->id) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        $data = InvoiceService::buildEmailData($order);
        return response()->json(['data' => $data]);
    }

    public function sendInvoiceEmail(Request $request, Order $order)
    {
        $user = $request->user();

        if ((int)$order->user_id !== (int)$user->id) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        // ✅ anti spam
        if ($order->invoice_emailed_at) {
            return response()->json([
                'message' => 'Invoice sudah pernah dikirim',
                'invoice_emailed_at' => $order->invoice_emailed_at,
            ], 200);
        }

        $data = InvoiceService::buildEmailData($order);
        Mail::to($user->email)->send(new InvoiceMail($data));

        $order->forceFill(['invoice_emailed_at' => now()])->save();

        Notification::create([
            'user_id' => $user->id,
            'title'   => 'Invoice terkirim',
            'message' => 'Invoice untuk Order #'.$order->id.' telah dikirim ke '.$user->email,
            'type'    => 'invoice',
            'is_read' => false,
        ]);

        return response()->json(['message' => 'Invoice email sent']);
    }

    public function updateStatus(Request $request, Order $order)
    {
        $validated = $request->validate([
            'order_status' => ['required', 'in:pending,processing,shipped,delivered,cancelled,returned'],
        ]);

        $old = $order->order_status;
        $order->order_status = $validated['order_status'];
        $order->save();

        // ✅ jangan kirim invoice dari sini
        return response()->json([
            'message' => 'Order status updated',
            'from'    => $old,
            'to'      => $order->order_status,
        ]);
    }
}
