<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\ProductBranchStock;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class OrderController extends Controller
{
    public function index(Request $request)
{
    $status = $request->query('status');

    $query = Order::query()
        ->with([
            'user:id,name',        // penting: supaya user.name ikut
            'branch:id,name',      // cabang
        ])
        ->select([
            'id',
            'invoice_number',
            'user_id',
            'branch_id',
            'customer_name_manual',
            'payment_method',
            'payment_status',
            'order_status',
            'total_amount',
            'created_at',
        ])
        ->latest('id');

    if ($status) {
        $query->where('order_status', $status);
    }

    return response()->json($query->paginate(10));
}

    public function show(Order $order)
    {
        $order->load('user', 'branch', 'items.product', 'items.variant');
        return response()->json($order);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'user_id' => 'required|exists:users,id',
            'branch_id' => 'required|exists:branches,id',
            'customer_name_manual' => 'nullable|string|max:100',
            'delivery_address' => 'required|string',
            'delivery_recipient_name' => 'required|string|max:100',
            'delivery_phone' => 'required|string|max:20',
            'total_amount' => 'required|numeric',
            'total_hpp' => 'required|numeric',
            'shipping_cost' => 'required|numeric',
            'payment_method' => 'nullable|string',
            'payment_proof' => 'nullable|string',
            'payment_status' => 'required|in:pending,paid,failed,expired,awaiting_verification',
            'order_status' => 'required|in:pending,processing,shipped,delivered,cancelled,returned',
            'shipping_courier' => 'nullable|string',
            'shipping_service' => 'nullable|string',
            'shipping_tracking_number' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.variant_id' => 'required|exists:product_variants,id',
            'items.*.product_name' => 'required|string',
            'items.*.variant_name' => 'required|string',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.price' => 'required|numeric',
            'items.*.hpp' => 'required|numeric',
            'delivery_postal_code' => 'nullable|string|max:10',
        ]);

        DB::transaction(function () use ($data) {
            // Buat invoice number unik
            $invoiceNumber = 'INV/' . date('Ymd') . '/' . $data['user_id'] . '/' . rand(1000, 9999);
            while (Order::where('invoice_number', $invoiceNumber)->exists()) {
                $invoiceNumber = 'INV/' . date('Ymd') . '/' . $data['user_id'] . '/' . rand(1000, 9999);
            }

            // Buat order
            $order = Order::create([
                'user_id' => $data['user_id'],
                'customer_name_manual' => $data['customer_name_manual'] ?? null,
                'branch_id' => $data['branch_id'],
                'delivery_address' => $data['delivery_address'],
                'delivery_recipient_name' => $data['delivery_recipient_name'],
                'delivery_phone' => $data['delivery_phone'],
                'invoice_number' => $invoiceNumber,
                'total_amount' => $data['total_amount'],
                'total_hpp' => $data['total_hpp'],
                'shipping_cost' => $data['shipping_cost'],
                'payment_method' => $data['payment_method'] ?? null,
                'payment_proof' => $data['payment_proof'] ?? null,
                'payment_status' => $data['payment_status'],
                'order_status' => $data['order_status'],
                'shipping_courier' => $data['shipping_courier'] ?? null,
                'shipping_service' => $data['shipping_service'] ?? null,
                'shipping_tracking_number' => $data['shipping_tracking_number'] ?? null,
                'delivery_postal_code' => $data['delivery_postal_code'] ?? null,
            ]);

            // Buat order items dan kurangi stok
            foreach ($data['items'] as $item) {
                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $item['product_id'],
                    'variant_id' => $item['variant_id'],
                    'product_name' => $item['product_name'],
                    'variant_name' => $item['variant_name'],
                    'quantity' => $item['quantity'],
                    'price' => $item['price'],
                    'hpp' => $item['hpp'],
                ]);

                // Kurangi stok di product_branch_stock
                $stock = ProductBranchStock::where('branch_id', $data['branch_id'])
                    ->where('product_id', $item['product_id'])
                    ->where('variant_id', $item['variant_id'])
                    ->first();

                if ($stock) {
                    $newStock = $stock->stock - $item['quantity'];
                    if ($newStock < 0) {
                        throw new \Exception("Stok tidak cukup untuk produk {$item['product_name']} variant {$item['variant_name']}");
                    }
                    $stock->update(['stock' => $newStock]);
                } else {
                    throw new \Exception("Stok tidak ditemukan untuk produk {$item['product_name']} variant {$item['variant_name']}");
                }
            }
        });

        return response()->json(['message' => 'Order berhasil dibuat dan stok dikurangi'], 201);
    }

    public function update(Request $request, Order $order)
    {
        $data = $request->validate([
            'order_status' => 'sometimes|in:pending,processing,shipped,delivered,cancelled,returned',
            'payment_status' => 'sometimes|in:pending,paid,failed,expired,awaiting_verification',
            'shipping_tracking_number' => 'nullable|string',
        ]);

        $order->update($data);
        return response()->json($order);
    }

    public function destroy(Order $order)
    {
        $order->delete();
        return response()->json(['message' => 'Order deleted']);
    }
}