<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Notification;
use Illuminate\Http\Request;

class NotificationController extends Controller
{
    // ✅ LIST NOTIF USER YANG LOGIN (bukan semua notif)
    public function index(Request $request)
    {
        $userId = $request->user()->id;

        $data = Notification::where('user_id', $userId)
            ->latest('id')
            ->paginate(10);

        return response()->json($data);
    }

    // ✅ UNREAD COUNT
    public function unreadCount(Request $request)
    {
        $userId = $request->user()->id;

        $count = Notification::where('user_id', $userId)
            ->where('is_read', 0)
            ->count();

        return response()->json(['count' => $count]);
    }

    // ✅ MARK ONE AS READ
    public function markRead(Request $request, Notification $notification)
    {
        // biar user lain gak bisa baca notif orang
        if ($notification->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        $notification->update(['is_read' => 1]);

        return response()->json(['message' => 'Notification marked as read']);
    }

    // ✅ MARK ALL AS READ
    public function markAllRead(Request $request)
    {
        $userId = $request->user()->id;

        Notification::where('user_id', $userId)
            ->where('is_read', 0)
            ->update(['is_read' => 1]);

        return response()->json(['message' => 'All notifications marked as read']);
    }

    // ✅ CREATE (admin/system) - tetap boleh
    public function store(Request $request)
    {
        $validated = $request->validate([
            'user_id' => ['required', 'integer', 'exists:users,id'],
            'title'   => ['required', 'string', 'max:255'],
            'message' => ['required', 'string'],
            'type'    => ['nullable', 'string', 'max:50'],
            'is_read' => ['nullable', 'boolean'],
        ]);

        $notification = Notification::create($validated);

        return response()->json($notification, 201);
    }

    // ✅ SHOW (hanya owner)
    public function show(Request $request, Notification $notification)
    {
        if ($notification->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        return response()->json($notification);
    }

    // ✅ UPDATE (hanya owner / admin)
    public function update(Request $request, Notification $notification)
    {
        if ($notification->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        $validated = $request->validate([
            'title'   => ['sometimes', 'string', 'max:255'],
            'message' => ['sometimes', 'string'],
            'type'    => ['nullable', 'string', 'max:50'],
            'is_read' => ['nullable', 'boolean'],
        ]);

        $notification->update($validated);

        return response()->json($notification);
    }

    // ✅ DELETE (hanya owner)
    public function destroy(Request $request, Notification $notification)
    {
        if ($notification->user_id !== $request->user()->id) {
            return response()->json(['message' => 'Forbidden'], 403);
        }

        $notification->delete();
        return response()->json(['message' => 'Notification deleted']);
    }
}
