<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Order;

class CustomerOrderController extends Controller
{
    public function myOrders(Request $request)
    {
        $user = $request->user();

        $orders = Order::with(['items.product.images'])
            ->where('user_id', $user->id)
            ->latest()
            ->get()
            ->map(function ($order) {

                $item = $order->items->first();
                $product = $item?->product;

                $imageModel = $product?->images
                    ->where('is_cover', 1)
                    ->first()
                    ?? $product?->images->first();

                $raw = $imageModel?->path ?? $imageModel?->image_url ?? null;

                $imageUrl = $this->publicImageUrl($raw);

                // ✅ TAMBAHAN: buat url bukti pembayaran yang benar
                $proofRaw =
                    $order->payment_proof
                    ?? $order->payment_proof_path
                    ?? $order->proof_payment
                    ?? $order->transfer_proof
                    ?? $order->proof
                    ?? null;

                $paymentProofUrl = $this->publicPaymentProofUrl($proofRaw);

                return [
                    'id' => $order->id,
                    'invoice' => $order->invoice_number,
                    'total' => (float) $order->total_amount,
                    'product_name' => $product?->name ?? '-',
                    'image' => $imageUrl,

                    'status' => $this->status($order),
                    'status_label' => $this->statusLabel($order),

                    'biteship_order_id' => $order->biteship_order_id,
                    'shipping_tracking_number' => $order->shipping_tracking_number,
                    'shipping_courier' => $order->shipping_courier,
                    'shipping_service' => $order->shipping_service,

                    'delivery_postal_code' => $order->delivery_postal_code ?? null,

                    'order_status' => $order->order_status,
                    'payment_status' => $order->payment_status,
                    // kalau ada di tabel, kirim juga biar FE bisa baca
                    'payment_method' => $order->payment_method ?? null,
                    'payment_proof' => $order->payment_proof ?? null,
                    'payment_proof_path' => $order->payment_proof_path ?? null,

                    // ✅ TAMBAHAN: FE enak tinggal pakai field ini
                    'payment_proof_url' => $paymentProofUrl,

                    'date' => $order->created_at->format('d F Y'),
                ];
            });

        return response()->json($orders);
    }

    private function publicImageUrl($value): ?string
    {
        if (!$value) return null;

        $s = trim((string)$value);
        if ($s === '') return null;

        if (preg_match('/^https?:\/\//i', $s)) {
            $s = preg_replace('#/api/uploads/#i', '/uploads/', $s);
            $s = preg_replace('#/api/uplouds/#i', '/uploads/', $s);
            return $s;
        }

        $s = str_replace('\\', '/', $s);

        $s = preg_replace('#^uplouds/#i', 'uploads/', $s);
        $s = preg_replace('#^/uplouds/#i', '/uploads/', $s);

        $s = preg_replace('#^public/#i', '', $s);
        $s = preg_replace('#^/public/#i', '/', $s);

        $s = preg_replace('#^storage/app/public/#i', '', $s);
        $s = preg_replace('#^/storage/app/public/#i', '/', $s);

        $s = preg_replace('#^storage/#i', '', $s);
        $s = preg_replace('#^/storage/#i', '/', $s);

        $s = preg_replace('#^/api/#i', '/', $s);

        $s = ltrim($s, '/');

        if (!preg_match('#^uploads/#i', $s)) {
            $s = 'uploads/' . $s;
        }

        return asset($s);
    }

    // ✅ TAMBAHAN (TIDAK MENGGANGGU KODE LAMA):
    // helper untuk bukti pembayaran yang disimpan via Storage disk public:
    // storage/app/public/payment-proofs/xxx -> public/storage/payment-proofs/xxx
    // URL publik: /storage/payment-proofs/xxx
    private function publicPaymentProofUrl($value): ?string
    {
        if (!$value) return null;

        $s = trim((string) $value);
        if ($s === '') return null;

        // kalau sudah url full, langsung pakai
        if (preg_match('/^https?:\/\//i', $s)) {
            return $s;
        }

        $s = str_replace('\\', '/', $s);

        // kalau db nyimpan "/storage/payment-proofs/xxx"
        $s = preg_replace('#^/?storage/#i', '', $s);
        $s = preg_replace('#^/?public/storage/#i', '', $s);

        // kalau db nyimpan "storage/app/public/payment-proofs/xxx"
        $s = preg_replace('#^/?storage/app/public/#i', '', $s);

        $s = ltrim($s, '/'); // jadi: payment-proofs/xxx

        // pastikan selalu masuk folder payment-proofs
        if (!preg_match('#^payment-proofs/#i', $s)) {
            // kalau dia nyimpen "proof_1.jpg" doang, tetap paksa masuk folder
            $s = 'payment-proofs/' . $s;
        }

        // hasil final: APP_URL/storage/payment-proofs/xxx
        return asset('storage/' . $s);
    }

    // ✅ helper: cek apakah user sudah upload bukti
    private function hasProof(Order $order): bool
    {
        $proof =
            $order->payment_proof
            ?? $order->payment_proof_path
            ?? $order->proof_payment
            ?? $order->transfer_proof
            ?? $order->proof
            ?? null;

        return !empty($proof);
    }

    private function status(Order $order)
    {
        // kalau sudah masuk alur order status, itu menang
        if (in_array($order->order_status, [
            'processing', 'shipped', 'delivered', 'cancelled', 'returned'
        ])) {
            return match ($order->order_status) {
                'processing' => 'PROCESSING',
                'shipped' => 'SHIPPED',
                'delivered' => 'COMPLETED',
                'cancelled', 'returned' => 'FAILED',
                default => 'PROCESSING',
            };
        }

        // ✅ PAYMENT LOGIC LURUS:
        // - paid => COMPLETED
        // - failed/expired => FAILED
        // - awaiting_verification => VERIFICATION
        // - pending => tergantung ada bukti atau tidak:
        //      pending + proof => VERIFICATION
        //      pending tanpa proof => UNPAID (belum bayar)
        // - unpaid => UNPAID
        return match ($order->payment_status) {
            'paid' => 'COMPLETED',
            'failed', 'expired' => 'FAILED',
            'awaiting_verification' => 'VERIFICATION',
            'pending' => $this->hasProof($order) ? 'VERIFICATION' : 'UNPAID',
            'unpaid' => 'UNPAID',
            default => 'UNPAID',
        };
    }

    private function statusLabel(Order $order)
    {
        if (in_array($order->order_status, [
            'processing', 'shipped', 'delivered', 'cancelled', 'returned'
        ])) {
            return match ($order->order_status) {
                'processing' => 'DIPROSES',
                'shipped' => 'DIKIRIM',
                'delivered' => 'SELESAI',
                'cancelled' => 'DIBATALKAN',
                'returned' => 'DIKEMBALIKAN',
                default => 'DIPROSES',
            };
        }

        return match ($order->payment_status) {
            'paid' => 'SELESAI',
            'failed' => 'GAGAL',
            'expired' => 'KADALUARSA',
            'awaiting_verification' => 'MENUNGGU VERIFIKASI',
            'pending' => $this->hasProof($order) ? 'MENUNGGU VERIFIKASI' : 'BELUM BAYAR',
            'unpaid' => 'BELUM BAYAR',
            default => 'BELUM BAYAR',
        };
    }
}
