<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CategoryController extends Controller
{
    // GET /api/categories
    public function index(Request $request)
    {
        $q = $request->q;

        $categories = Category::when($q, function ($query) use ($q) {
                $query->where('name', 'like', "%$q%")
                      ->orWhere('slug', 'like', "%$q%");
            })
            ->orderByDesc('id')
            ->paginate(20);

        return response()->json($categories);
    }

    // GET /api/categories/{id}
    public function show($id)
    {
        $category = Category::findOrFail($id);
        return response()->json(['data' => $category]);
    }

    // GET /api/category/{slug}
    public function showBySlug($slug)
    {
        $category = Category::where('slug', $slug)->firstOrFail();
        return response()->json(['data' => $category]);
    }

    // POST /api/categories
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:100',
            'slug' => 'nullable|string|max:100|unique:categories,slug',
            'icon_url' => 'nullable|string',
        ]);

        $slug = $request->slug ?: Str::slug($request->name);

        // pastikan slug unik
        $baseSlug = $slug;
        $i = 2;
        while (Category::where('slug', $slug)->exists()) {
            $slug = "{$baseSlug}-{$i}";
            $i++;
        }

        $category = Category::create([
            'name' => $request->name,
            'slug' => $slug,
            'icon_url' => $request->icon_url,
        ]);

        return response()->json([
            'message' => 'Kategori berhasil dibuat',
            'data' => $category
        ], 201);
    }

    // PUT /api/categories/{id}
    public function update(Request $request, $id)
    {
        $category = Category::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:100',
            'slug' => 'nullable|string|max:100|unique:categories,slug,' . $id,
            'icon_url' => 'nullable|string',
        ]);

        $slug = $request->slug ?: Str::slug($request->name);

        // unik kecuali dirinya sendiri
        $baseSlug = $slug;
        $i = 2;
        while (
            Category::where('slug', $slug)
                ->where('id', '!=', $id)
                ->exists()
        ) {
            $slug = "{$baseSlug}-{$i}";
            $i++;
        }

        $category->update([
            'name' => $request->name,
            'slug' => $slug,
            'icon_url' => $request->icon_url,
        ]);

        return response()->json([
            'message' => 'Kategori berhasil diupdate',
            'data' => $category
        ]);
    }

    // DELETE /api/categories/{id}
    public function destroy($id)
    {
        Category::findOrFail($id)->delete();
        return response()->json(['message' => 'Kategori berhasil dihapus']);
    }
}
