<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Order;
use Illuminate\Http\Request;

class BiteshipWebhookController extends Controller
{
    public function handle(Request $request)
    {
        $payload   = $request->getContent();
        $signature = $request->header('X-Biteship-Signature');
        $secret    = (string) config('services.biteship.webhook_secret', '');

        if ($secret && !$this->isValidSignature($payload, $signature, $secret)) {
            return response()->json(['message' => 'Invalid signature'], 401);
        }

        $data = $request->all();

        $orderId        = data_get($data, 'order_id');
        $waybillId      = data_get($data, 'waybill_id');
        $status         = data_get($data, 'status');
        $trackingNumber = data_get($data, 'tracking_id');

        if (!$orderId) {
            return response()->json(['message' => 'order_id is required'], 422);
        }

        $order = Order::query()->where('biteship_order_id', $orderId)->first();
        if (!$order) {
            return response()->json(['message' => 'Order not found'], 404);
        }

        if ($waybillId) $order->biteship_waybill_id = $waybillId;
        if ($trackingNumber) $order->shipping_tracking_number = $trackingNumber;
        if ($status) $order->order_status = $this->mapStatus($status, (string) $order->order_status);

        $order->save();

        return response()->json(['message' => 'Webhook processed']);
    }

    private function isValidSignature(string $payload, ?string $signature, string $secret): bool
    {
        if (!$signature) return false;

        $expected = hash_hmac('sha256', $payload, $secret);
        return hash_equals($expected, $signature);
    }

    private function mapStatus(string $status, string $fallback): string
    {
        $status = strtolower(trim($status));

        return match ($status) {
            'confirmed', 'allocated' => 'processing',
            'picked_up', 'in_transit' => 'shipped',
            'delivered' => 'delivered',
            'cancelled', 'rejected' => 'cancelled',
            default => $fallback,
        };
    }
}
