<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\BiteshipOrderRequest;
use App\Http\Requests\BiteshipRatesRequest;
use App\Models\Order;
use App\Services\BiteshipService;
use Illuminate\Http\Request;
use Illuminate\Http\Client\Response as HttpResponse;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class BiteshipController extends Controller
{
    public function __construct(private BiteshipService $biteship) {}

    private function baseUrl(): string
    {
        return rtrim((string) config('services.biteship.base_url', 'https://api.biteship.com'), '/');
    }

    private function apiKey(): string
    {
        return trim((string) config('services.biteship.key', ''));
    }

    private function headersBearer(): array
    {
        return [
            'Authorization' => 'Bearer ' . $this->apiKey(),
            'Accept'        => 'application/json',
            'Content-Type'  => 'application/json',
        ];
    }

    private function headersRaw(): array
    {
        return [
            'Authorization' => $this->apiKey(),
            'Accept'        => 'application/json',
            'Content-Type'  => 'application/json',
        ];
    }

    /**
     * Helper: normalize response dari service yang bisa return:
     * - Illuminate\Http\Client\Response
     * - array
     */
    private function respondFromService($res)
    {
        if ($res instanceof HttpResponse) {
            return response()->json($res->json(), $res->status());
        }

        if (is_array($res)) {
            // kalau array punya status code, pakai itu
            $status = (int) ($res['status'] ?? 200);

            // kalau array adalah wrapper {success, data, ...} biarkan apa adanya
            return response()->json($res, $status);
        }

        // fallback kalau return tidak jelas
        return response()->json([
            'success' => false,
            'message' => 'Invalid response from service.',
        ], 500);
    }

    /**
     * Ambil courier aktif dari Biteship (yang di-enable di dashboard).
     * Cache 6 jam.
     * Output: "jne,sicepat,anteraja,gojek,..."
     */
    private function getActiveCouriersCsv(): string
    {
        return Cache::remember('biteship_active_couriers_csv', now()->addHours(6), function () {
            $url = $this->baseUrl() . '/v1/couriers';

            // coba Bearer
            $res = Http::withHeaders($this->headersBearer())
                ->timeout(30)
                ->get($url);

            // fallback raw api key
            if (!$res->successful()) {
                $res = Http::withHeaders($this->headersRaw())
                    ->timeout(30)
                    ->get($url);
            }

            if (!$res->successful()) {
                Log::warning('Biteship get couriers failed', [
                    'status' => $res->status(),
                    'body'   => $res->json(),
                ]);

                $fallback = (string) config('services.biteship.default_couriers', 'jne,sicepat,anteraja');
                return trim($fallback) ?: 'jne,sicepat,anteraja';
            }

            $json = $res->json();

            $list =
                data_get($json, 'couriers') ??
                data_get($json, 'data.couriers') ??
                data_get($json, 'data') ??
                [];

            if (!is_array($list)) $list = [];

            $codes = [];
            foreach ($list as $c) {
                $code = strtolower(trim((string) (data_get($c, 'courier_code') ?? data_get($c, 'code') ?? '')));
                $active = data_get($c, 'is_active');

                // kalau is_active tidak ada → anggap aktif
                if ($active === null) $active = true;

                if ($code && $active) $codes[] = $code;
            }

            $codes = array_values(array_unique($codes));

            if (count($codes) === 0) {
                $fallback = (string) config('services.biteship.default_couriers', 'jne,sicepat,anteraja');
                return trim($fallback) ?: 'jne,sicepat,anteraja';
            }

            return implode(',', $codes);
        });
    }

    // =========================
    // POST /api/shipping/rates
    // =========================
    public function rates(BiteshipRatesRequest $request)
    {
        $payload = $request->validated();

        // normalize
        $payload['origin_postal_code'] = (string) ($payload['origin_postal_code'] ?? '');
        $payload['destination_postal_code'] = (string) ($payload['destination_postal_code'] ?? '');

        // kalau FE kirim kosong / tidak kirim → pakai yang aktif di dashboard
        $couriers = trim((string) ($payload['couriers'] ?? ''));
        if ($couriers === '') {
            $payload['couriers'] = $this->getActiveCouriersCsv();
        }

        // guard minimal biar tidak 40001002
        if (!$payload['origin_postal_code'] || !$payload['destination_postal_code']) {
            return response()->json([
                'success' => false,
                'message' => 'origin_postal_code & destination_postal_code wajib diisi.',
            ], 422);
        }

        if (empty($payload['items']) || !is_array($payload['items'])) {
            return response()->json([
                'success' => false,
                'message' => 'items wajib diisi (array).',
            ], 422);
        }

        // call service
        $res = $this->biteship->rates($payload);

        // ✅ fix: jangan panggil json() kalau array
        return $this->respondFromService($res);
    }

    // OPTIONAL: cek couriers aktif (buat debug)
    // GET /api/shipping/active-couriers
    public function activeCouriers()
    {
        return response()->json([
            'success' => true,
            'couriers' => $this->getActiveCouriersCsv(),
        ]);
    }

    // =========================
    // POST /api/shipping/orders
    // =========================
    public function createOrder(BiteshipOrderRequest $request)
    {
        $payload = $request->validated();
        $payload['origin_note'] = $payload['origin_note'] ?? (string) config('services.biteship.origin_note', '');

        $res = $this->biteship->createOrder($payload);

        return $this->respondFromService($res);
    }

    // =========================
    // POST /api/shipping/orders/{order}/biteship
    // =========================
    public function createOrderFromLocal(Request $request, Order $order)
    {
        if (!empty($order->biteship_order_id) || !empty($order->shipping_tracking_number)) {
            return response()->json([
                'success' => false,
                'message' => 'Biteship order / resi sudah ada untuk order ini.',
            ], 409);
        }

        $input = $request->input('courier', $request->all());

        $v = Validator::make($input, [
            'company' => 'required|string',
            'type' => 'required|string',
            'destination_postal_code' => 'required|string|min:5',
            'insurance' => 'nullable|numeric|min:0',
            'weight' => 'nullable|numeric|min:1',
        ]);

        if ($v->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validasi gagal.',
                'errors' => $v->errors(),
            ], 422);
        }

        $data = $v->validated();

        $order->load(['items']);
        if ($order->items->count() === 0) {
            return response()->json([
                'success' => false,
                'message' => 'Order tidak punya item. Tidak bisa buat pengiriman.',
            ], 422);
        }

        // origin config
        $originPostalCode  = (string) config('services.biteship.origin_postal_code', '');
        $originName        = (string) config('services.biteship.origin_contact_name', '');
        $originPhone       = (string) config('services.biteship.origin_contact_phone', '');
        $originAddress     = (string) config('services.biteship.origin_address', '');
        $originNote        = (string) config('services.biteship.origin_note', '');

        $missing = [];
        if (!$originPostalCode) $missing[] = 'BITESHIP_ORIGIN_POSTAL_CODE';
        if (!$originName) $missing[] = 'BITESHIP_ORIGIN_CONTACT_NAME';
        if (!$originPhone) $missing[] = 'BITESHIP_ORIGIN_CONTACT_PHONE';
        if (!$originAddress) $missing[] = 'BITESHIP_ORIGIN_ADDRESS';

        if ($missing) {
            return response()->json([
                'success' => false,
                'message' => 'Konfigurasi origin Biteship belum lengkap.',
                'missing' => $missing,
            ], 500);
        }

        // normalize courier
        $company = strtolower(trim($data['company']));
        $type    = strtolower(trim($data['type']));
        $typeMap = ['reguler' => 'reg', 'regular' => 'reg'];
        if (isset($typeMap[$type])) $type = $typeMap[$type];

        // map items local -> biteship
        $items = [];
        foreach ($order->items as $it) {
            $items[] = [
                'name'     => (string) ($it->product_name ?? 'Item'),
                'quantity' => max(1, (int) ($it->quantity ?? 1)),
                'value'    => max(0, (int) ($it->price ?? 0)),
                'weight'   => max(1, (int) ($it->weight ?? 200)),
            ];
        }

        $weight = (int) ($data['weight'] ?? collect($items)->sum('weight'));
        if ($weight < 1) $weight = 1000;

        $payload = [
            'reference_id' => $order->invoice_number ?? ('ORDER-' . $order->id),

            'origin_contact_name'  => $originName,
            'origin_contact_phone' => $originPhone,
            'origin_address'       => $originAddress,
            'origin_postal_code'   => $originPostalCode,
            'origin_note'          => $originNote,

            'destination_contact_name'  => $order->delivery_recipient_name ?: ($order->customer_name_manual ?: 'Customer'),
            'destination_contact_phone' => $order->delivery_phone ?: '080000000000',
            'destination_address'       => $order->delivery_address ?: '-',
            'destination_postal_code'   => $data['destination_postal_code'],

            'courier_company' => $company,
            'courier_type'    => $type,

            'delivery_type' => 'now',
            'items' => $items,
            'weight' => $weight,
            'insurance' => (int) ($data['insurance'] ?? 0),
            'order_note' => 'Order #' . ($order->invoice_number ?? $order->id),
        ];

        $url = $this->baseUrl() . '/v1/orders';

        // Bearer dulu, fallback raw
        $res = Http::withHeaders($this->headersBearer())->timeout(30)->post($url, $payload);
        if (!$res->successful()) {
            $res = Http::withHeaders($this->headersRaw())->timeout(30)->post($url, $payload);
        }

        if (!$res->successful()) {
            Log::warning('Biteship createOrderFromLocal failed', [
                'order_id' => $order->id,
                'status' => $res->status(),
                'response' => $res->json(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Biteship reject request.',
                'biteship_status' => $res->status(),
                'biteship' => $res->json(),
            ], $res->status());
        }

        $body = $res->json();

        $biteshipOrderId = data_get($body, 'id') ?? data_get($body, 'data.id');
        $trackingNumber  =
            data_get($body, 'courier.tracking_id') ??
            data_get($body, 'tracking_number') ??
            data_get($body, 'data.tracking_number');

        $update = [];
        if ($biteshipOrderId) $update['biteship_order_id'] = $biteshipOrderId;
        if ($trackingNumber)  $update['shipping_tracking_number'] = $trackingNumber;

        if ($update) $order->update($update);

        return response()->json([
            'success' => true,
            'message' => 'Biteship order created.',
            'biteship' => $body,
            'saved' => $update,
        ]);
    }

    // =========================
    // GET /api/shipping/orders/{orderId}
    // =========================
    public function track(string $orderId)
    {
        $res = $this->biteship->track($orderId);
        return $this->respondFromService($res);
    }
}
