<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;
use Illuminate\Validation\Rules\Password;
use Google\Client as GoogleClient;

class AuthController extends Controller
{
    public function register(Request $request)
    {
        $validated = $request->validate([
            'name'     => ['required','string','max:100'],
            'email'    => ['required','email','max:100','unique:users,email'],
            'password' => ['required', Password::min(8)],
            'phone'    => ['nullable','string','max:20'],
        ]);

        $user = User::create([
            'name'      => $validated['name'],
            'email'     => $validated['email'],
            'password'  => Hash::make($validated['password']),
            'phone'     => $validated['phone'] ?? null,
            'role'      => 'customer',   // default register customer
            'is_active' => 1,
        ]);

        $token = $user->createToken('api-token')->plainTextToken;

        return response()->json([
            'token' => $token,
            'user'  => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'role' => $user->role,
            ]
        ], 201);
    }

    public function login(Request $request)
    {
        $request->validate([
            'email' => ['required','email'],
            'password' => ['required','string'],
        ]);

        $user = User::where('email', $request->email)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            throw ValidationException::withMessages([
                'email' => ['Email atau password salah.'],
            ]);
        }

        if ((int) $user->is_active !== 1) {
            return response()->json(['message' => 'Akun nonaktif'], 403);
        }

        // WAJIB: hanya staff boleh login admin panel
        if (!$user->isStaff()) {
            return response()->json(['message' => 'Akun ini bukan staff admin.'], 403);
        }

        $token = $user->createToken('admin-token')->plainTextToken;

        return response()->json([
            'token' => $token,
            'user'  => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'role' => $user->role,
            ]
        ]);
    }

    public function me(Request $request)
    {
        return response()->json($request->user());
    }

    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();
        return response()->json(['message' => 'Logged out']);
    }

    public function googleLogin(Request $request)
{
    $validated = $request->validate([
        'id_token' => ['required','string'],
    ]);

    $client = new GoogleClient([
        'client_id' => env('GOOGLE_CLIENT_ID'),
    ]);

    $payload = $client->verifyIdToken($validated['id_token']);
    if (!$payload) {
        return response()->json(['message' => 'Google token invalid'], 401);
    }

    $googleId = $payload['sub'];
    $email    = $payload['email'] ?? null;
    $name     = $payload['name'] ?? 'Google User';

    if (!$email) {
        return response()->json(['message' => 'Email Google tidak tersedia'], 422);
    }

    // cari user by google_id atau email
    $user = User::where('google_id', $googleId)
        ->orWhere('email', $email)
        ->first();

    if (!$user) {
        $user = User::create([
            'name'      => $name,
            'email'     => $email,
            'password'  => Hash::make(str()->random(32)), // random, karena login via google
            'role'      => 'customer',
            'is_active' => 1,
            'google_id' => $googleId,
        ]);
    } else {
        // pastikan google_id tersimpan
        if (!$user->google_id) {
            $user->google_id = $googleId;
            $user->save();
        }
        if ((int)$user->is_active !== 1) {
            return response()->json(['message' => 'Akun nonaktif'], 403);
        }
    }

    $token = $user->createToken('api-token')->plainTextToken;

    return response()->json([
        'token' => $token,
        'user'  => [
            'id' => $user->id,
            'name' => $user->name,
            'email' => $user->email,
            'role' => $user->role,
        ]
    ]);
}
}