<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderItem;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AnalyticsController extends Controller
{
    public function index(Request $request)
    {
        $validated = $request->validate([
            'range' => 'nullable|in:7,30',
            'from'  => 'nullable|date_format:Y-m-d',
            'to'    => 'nullable|date_format:Y-m-d',
        ]);

        // Tentukan periode
        [$start, $end] = $this->resolveDateRange($request);

        // Base filter: periode + (opsional) exclude cancelled untuk "sales performance"
        $baseOrders = Order::query()
            ->whereBetween('created_at', [$start, $end])
            ->whereNotIn('order_status', ['cancelled']);

        // Summary
        $summary = [
            // Revenue: hanya order paid (umum dipakai untuk “omzet real”)
            'revenue'     => (float) (clone $baseOrders)->where('payment_status', 'paid')->sum('total_amount'),
            'orders'      => (int) (clone $baseOrders)->count(),
            'paid_orders' => (int) (clone $baseOrders)->where('payment_status', 'paid')->count(),
            'customers'   => (int) (clone $baseOrders)->distinct('user_id')->count('user_id'),
        ];

        // Daily trend: group by DATE(created_at)
        $dailyRaw = (clone $baseOrders)
            ->selectRaw('DATE(created_at) as date')
            ->selectRaw('COUNT(*) as orders')
            ->selectRaw('SUM(CASE WHEN payment_status = "paid" THEN total_amount ELSE 0 END) as revenue')
            ->groupBy(DB::raw('DATE(created_at)'))
            ->orderBy('date')
            ->get()
            ->keyBy('date');

        // Fill missing dates (biar chart mulus walau ada hari kosong)
        $period = CarbonPeriod::create($start->copy()->startOfDay(), $end->copy()->startOfDay());
        $daily = [];
        foreach ($period as $date) {
            $d = $date->format('Y-m-d');
            $row = $dailyRaw->get($d);

            $daily[] = [
                'date'    => $d,
                'orders'  => (int) ($row->orders ?? 0),
                'revenue' => (float) ($row->revenue ?? 0),
            ];
        }

        // Top Products (paid only)
        $topProducts = OrderItem::query()
            ->join('orders', 'orders.id', '=', 'order_items.order_id')
            ->whereBetween('orders.created_at', [$start, $end])
            ->whereNotIn('orders.order_status', ['cancelled'])
            ->where('orders.payment_status', 'paid')
            ->selectRaw('order_items.product_id as product_id')
            ->selectRaw('order_items.product_name as name')
            ->selectRaw('SUM(order_items.quantity) as qty')
            ->selectRaw('SUM(order_items.price * order_items.quantity) as revenue')
            ->groupBy('order_items.product_id', 'order_items.product_name')
            ->orderByDesc('revenue')
            ->limit(10)
            ->get()
            ->map(fn($r) => [
                'product_id' => (int) $r->product_id,
                'name'       => $r->name,
                'qty'        => (int) $r->qty,
                'revenue'    => (float) $r->revenue,
            ]);

        // Top Branches (paid only)
        $topBranches = Order::query()
            ->leftJoin('branches', 'branches.id', '=', 'orders.branch_id')
            ->whereBetween('orders.created_at', [$start, $end])
            ->whereNotIn('orders.order_status', ['cancelled'])
            ->where('orders.payment_status', 'paid')
            ->selectRaw('COALESCE(branches.name, "Unknown Branch") as name')
            ->selectRaw('COUNT(*) as orders')
            ->selectRaw('SUM(orders.total_amount) as revenue')
            ->groupBy('branches.name')
            ->orderByDesc('revenue')
            ->limit(10)
            ->get()
            ->map(fn($r) => [
                'name'    => $r->name,
                'orders'  => (int) $r->orders,
                'revenue' => (float) $r->revenue,
            ]);

        // Payment status breakdown (buat donut chart)
        $paymentBreakdown = (clone $baseOrders)
            ->select('payment_status')
            ->selectRaw('COUNT(*) as total')
            ->groupBy('payment_status')
            ->get()
            ->map(fn($r) => [
                'status' => $r->payment_status,
                'total'  => (int) $r->total,
            ]);

        return response()->json([
            'data' => [
                'meta' => [
                    'from' => $start->toDateString(),
                    'to'   => $end->toDateString(),
                ],
                'summary'           => $summary,
                'daily'             => $daily,
                'top_products'      => $topProducts,
                'top_branches'      => $topBranches,
                'payment_breakdown' => $paymentBreakdown,
            ]
        ]);
    }

    private function resolveDateRange(Request $request): array
    {
        // Inclusive end: sampai akhir hari
        if ($request->filled('from') && $request->filled('to')) {
            $start = Carbon::createFromFormat('Y-m-d', $request->from)->startOfDay();
            $end   = Carbon::createFromFormat('Y-m-d', $request->to)->endOfDay();
            return [$start, $end];
        }

        $range = (int) ($request->get('range', 7));
        $end   = Carbon::now()->endOfDay();
        $start = Carbon::now()->subDays($range - 1)->startOfDay();

        return [$start, $end];
    }
}